/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.generators;

import java.io.IOException;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.SAXException;

import org.ametys.core.right.RightManager;
import org.ametys.plugins.forms.dao.FormEntryDAO;
import org.ametys.plugins.forms.helper.FormStatisticsHelper;
import org.ametys.plugins.forms.helper.FormStatisticsHelper.Option;
import org.ametys.plugins.forms.question.types.CheckBoxQuestionType;
import org.ametys.plugins.forms.question.types.ChoicesListQuestionType;
import org.ametys.plugins.forms.question.types.MatrixQuestionType;
import org.ametys.plugins.forms.repository.Form;
import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.plugins.repository.AmetysObjectResolver;

/**
 * Generate a stat map of a form's entries
 */
public class SummaryExportGenerator extends ServiceableGenerator
{
    /** The Ametys Object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The form statistics helper */
    protected FormStatisticsHelper _formStatsHelper;
    
    /** The right manager */
    protected RightManager _rightManager;
    
    /** The form entry dao */
    protected FormEntryDAO _formEntryDAO;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _formStatsHelper = (FormStatisticsHelper) smanager.lookup(FormStatisticsHelper.ROLE);
        _rightManager = (RightManager) smanager.lookup(RightManager.ROLE);
        _formEntryDAO = (FormEntryDAO) smanager.lookup(FormEntryDAO.ROLE);
    }
    
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        String formId = (String) request.get("id");
        
        contentHandler.startDocument();
        
        Form form = _resolver.resolveById(formId);
        _formEntryDAO.checkHandleDataRight(form);

        Map<String, Map<String, Map<String, Object>>> statsMap = _formStatsHelper.getStatsMap(form);
        
        AttributesImpl formAttrs = new AttributesImpl();
        formAttrs.addCDATAAttribute("id", form.getId());
        formAttrs.addCDATAAttribute("entries-count", String.valueOf(form.getEntries().size()));
        
        XMLUtils.startElement(contentHandler, "form", formAttrs);
        for (String questionId : statsMap.keySet())
        {
            _saxQuestionStats(form, statsMap, questionId);
        }
        XMLUtils.endElement(contentHandler, "form");
        
        contentHandler.endDocument();
    }
    
    /**
     * Sax a question stats for summary export
     * @param form current form
     * @param statsMap map with stats values
     * @param questionName the name for form of the question
     * @throws SAXException exception while saxing question
     */
    protected void _saxQuestionStats(Form form, Map<String, Map<String, Map<String, Object>>> statsMap, String questionName) throws SAXException
    {
        FormQuestion question = form.getQuestion(questionName);
        AttributesImpl questionAttrs = new AttributesImpl();
        questionAttrs.addCDATAAttribute("type", question.getType().getId());
        questionAttrs.addCDATAAttribute("title", question.getTitle());
        XMLUtils.startElement(contentHandler, "question", questionAttrs);
        
        if (question.getType() instanceof MatrixQuestionType)
        {
            _saxMatrixStats(statsMap, question);
        }
        else if (question.getType() instanceof ChoicesListQuestionType type)
        {
            _saxChoicesStats(statsMap, questionName);
        }
        else if (question.getType() instanceof CheckBoxQuestionType)
        {
            _saxBooleanStats(statsMap, questionName);
        }
        else
        {
            _saxDefaultQuestionStats(statsMap, questionName);
        }
        XMLUtils.endElement(contentHandler, "question");
    }

    /**
     * Sax a matrix question stats
     * @param statsMap map with stats values
     * @param question the question
     * @throws SAXException if a saxing error occurred
     */
    protected void _saxMatrixStats(Map<String, Map<String, Map<String, Object>>> statsMap, FormQuestion question) throws SAXException
    {
        MatrixQuestionType type = (MatrixQuestionType) question.getType();
        
        Map<String, Map<String, Object>> rows = statsMap.get(question.getNameForForm());
        XMLUtils.startElement(contentHandler, "values");
        for (String row: rows.keySet())
        {
            AttributesImpl rowAttrs = new AttributesImpl();
            rowAttrs.addCDATAAttribute("label", type.getRows(question).get(row));
            rowAttrs.addCDATAAttribute("value", row);
            XMLUtils.startElement(contentHandler, "options", rowAttrs);
            
            Map<String, Object> columns = rows.get(row);
            for (String  col :columns.keySet())
            {
                AttributesImpl colsAttrs = new AttributesImpl();
                colsAttrs.addCDATAAttribute("label", type.getColumns(question).get(col));
                colsAttrs.addCDATAAttribute("value", col);
                colsAttrs.addCDATAAttribute("count", String.valueOf(columns.get(col)));
                XMLUtils.createElement(contentHandler, "option", colsAttrs);
            }
            XMLUtils.endElement(contentHandler, "options");
        }
        XMLUtils.endElement(contentHandler, "values");
    }
    
    /**
     * Sax a choices list question stats
     * @param statsMap map with stats values
     * @param questionName the question name
     * @throws SAXException if a saxing error occurred
     */
    protected void _saxChoicesStats(Map<String, Map<String, Map<String, Object>>> statsMap, String questionName) throws SAXException
    {
        Map<String, Map<String, Object>> values = statsMap.get(questionName);
        XMLUtils.startElement(contentHandler, "values");
        
        Map<String, Object> options = values.get("values");
        for (String option : options.keySet())
        {
            AttributesImpl optAttrs = new AttributesImpl();
            optAttrs.addCDATAAttribute("label", ((Option) options.get(option)).label());
            optAttrs.addCDATAAttribute("count", String.valueOf(((Option) options.get(option)).count()));
            optAttrs.addCDATAAttribute("value", option);
            XMLUtils.createElement(contentHandler, "option", optAttrs);
        }
        XMLUtils.endElement(contentHandler, "values");
    }
    
    /**
     * Sax a boolean question stats
     * @param statsMap map with stats values
     * @param questionName the question name
     * @throws SAXException if a saxing error occurred
     */
    protected void _saxBooleanStats(Map<String, Map<String, Map<String, Object>>> statsMap, String questionName) throws SAXException
    {
        Map<String, Object> values = statsMap.get(questionName).get("values");
        AttributesImpl boolAttrs = new AttributesImpl();
        boolAttrs.addCDATAAttribute("true", String.valueOf(values.get("true")));
        boolAttrs.addCDATAAttribute("false", String.valueOf(values.get("false")));
        XMLUtils.createElement(contentHandler, "values", boolAttrs);
    }

    /**
     * Sax a default question stats
     * @param statsMap map with stats values
     * @param questionName the question name
     * @throws SAXException if a saxing error occurred
     */
    protected void _saxDefaultQuestionStats(Map<String, Map<String, Map<String, Object>>> statsMap, String questionName) throws SAXException
    {
        Map<String, Object> values = statsMap.get(questionName).get("values");
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("answered", String.valueOf(values.get("answered")));
        attrs.addCDATAAttribute("empty", String.valueOf(values.get("empty")));
        XMLUtils.createElement(contentHandler, "values", attrs);
    }
}
