/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.helper;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.core.ui.Callable;
import org.ametys.plugins.forms.dao.FormDAO;
import org.ametys.plugins.forms.question.FormQuestionType;
import org.ametys.plugins.forms.question.sources.ChoiceSourceType;
import org.ametys.plugins.forms.question.types.CheckBoxQuestionType;
import org.ametys.plugins.forms.question.types.ChoicesListQuestionType;
import org.ametys.plugins.forms.repository.Form;
import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.plugins.forms.repository.type.Rule.QuestionRuleType;
import org.ametys.plugins.forms.rights.FormsDirectoryRightAssignmentContext;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * Helper for getting rules properties
 */
public class QuestionRuleHelper implements Serviceable, Component
{
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
    }
    
    /**
     * Get all choices list and booleans questions as source for the rule
     * @param formId id of the current form
     * @param questionId id of the current question
     * @return a map containing the list of questions
     */
    @Callable (rights = FormDAO.HANDLE_FORMS_RIGHT_ID, rightContext = FormsDirectoryRightAssignmentContext.ID, paramIndex = 0)
    public Map<String, Object> getSourceQuestions(String formId, String questionId)
    {
        Map<String, Object> results = new HashMap<>();
        Form form = _resolver.resolveById(formId);
        List<FormQuestion> questions = form.getQuestions()
            .stream()
            .filter(question -> this._isChoiceListWithNoTooManyOptions(question) || question.getType() instanceof CheckBoxQuestionType) // Question must be a choice list with not too many options
            .filter(question -> !question.isModifiable()) // Question must be not modifiable
            .filter(question -> !question.getId().equals(questionId)) // Remove the given question from the list
            .collect(Collectors.toList());
        List<Object> sources = new ArrayList<>();
        for (FormQuestion question : questions)
        {
            Map<String, String> properties = new HashMap<>();
            properties.put("label", question.getTitle());
            properties.put("id", question.getId());
            sources.add(properties);
        }
        results.put("data", sources);
        
        return results;
    }
    
    private boolean _isChoiceListWithNoTooManyOptions(FormQuestion question)
    {
        FormQuestionType type = question.getType();
        if (type instanceof ChoicesListQuestionType cLType)
        {
            ChoiceSourceType sourceType = cLType.getSourceType(question);
            return !sourceType.remoteData();
        }
        
        return false;
    }
    
    /**
     * Get all the available options for the selected question
     * @param questionId id of the selected question
     * @return a map containing the list of all the options as map (label for display and unique value)
     */
    @Callable (rights = FormDAO.HANDLE_FORMS_RIGHT_ID, rightContext = FormsDirectoryRightAssignmentContext.ID, paramIndex = 0)
    public Map<String, Object> getSourceOptions(String questionId)
    {
        Map<String, Object> results = new HashMap<>();
        FormQuestion formQuestion = _resolver.resolveById(questionId);
        List<Object> options = new ArrayList<>();
        if (formQuestion.getType() instanceof ChoicesListQuestionType type && !type.getSourceType(formQuestion).remoteData())
        {
            Map<String, I18nizableText> choices = type.getOptions(formQuestion);
            for (String optionValue : choices.keySet())
            {
                Map<String, Object> choicesToJson = new HashMap<>();
                choicesToJson.put("label", choices.get(optionValue));
                choicesToJson.put("value", optionValue);
                options.add(choicesToJson);
            }
            if (type.hasOtherOption(formQuestion))
            {
                Map<String, Object> choicesToJson = new HashMap<>();
                choicesToJson.put("label", new I18nizableText("plugin.forms", "PLUGINS_FORMS_DISPLAY_OTHER_OPTION_COMBOBOX"));
                choicesToJson.put("value", ChoicesListQuestionType.OTHER_OPTION_VALUE);
                options.add(choicesToJson);
            }
        }
        else
        {
            options.add(Map.of(
                    "label", new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTION_RULES_OPTION_CHECKED"),
                    "value", "true"
                    ));
            
            options.add(Map.of(
                    "label", new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTION_RULES_OPTION_UNCHECKED"),
                    "value", "false"
                    ));
        }
        results.put("data", options);
        return results;
    }
    
    /**
     * Get the actions available to apply to this question 
     * @return a list of actions (currently hide or show)
     */
    @Callable
    public Map<String, Object> getRuleActions()
    {
        Map<String, Object> results = new HashMap<>();
        
        List<Object> actions = new ArrayList<>();
        
        actions.add(Map.of(
            "label", QuestionRuleType.HIDE.getLabel(),
            "value", QuestionRuleType.HIDE.name()
        ));
        
        actions.add(Map.of(
            "label", QuestionRuleType.SHOW.getLabel(),
            "value", QuestionRuleType.SHOW.name()
        ));
        
        results.put("data", actions);
        return results;
    }
}
