/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.question.sources;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.content.indexing.solr.SolrFieldNames;
import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeEnumerator;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.cms.search.Sort;
import org.ametys.cms.search.Sort.Order;
import org.ametys.cms.search.content.ContentSearcherFactory;
import org.ametys.cms.search.ui.model.SearchUIModel;
import org.ametys.cms.search.ui.model.SearchUIModelExtensionPoint;
import org.ametys.plugins.forms.enumerators.LazyEnumerator;
import org.ametys.plugins.forms.helper.FormElementDefinitionHelper;
import org.ametys.plugins.forms.question.types.ChoicesListQuestionType;
import org.ametys.plugins.forms.repository.FormEntry;
import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.StaticEnumerator;
import org.ametys.runtime.model.ViewElement;
import org.ametys.runtime.model.ViewItem;
import org.ametys.runtime.model.type.ModelItemTypeConstants;
import org.ametys.runtime.parameter.DefaultValidator;

/**
 * Class for creating users choice list from a population
 */
public class TableRefSourceType extends AbstractSourceType
{
    /** Constant for element definition name of the table ref id*/
    public static final String ATTRIBUTE_TABLE_REF_ID = "table-ref-id";
    
    /** Map of ModelItems specific to UsersSourceType */
    protected Map<String, ModelItem> _tableRefSourceItems;
    
    /** The content types extension point */
    protected ContentTypeExtensionPoint _cTypesEP;
    
    /** The searcher factory */
    protected ContentSearcherFactory _searcherFactory;
    
    /** The search UI model extension point */
    protected SearchUIModelExtensionPoint _searchUIModelExtensionPoint;

    private ServiceManager _manager;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _searcherFactory = (ContentSearcherFactory) manager.lookup(ContentSearcherFactory.ROLE);
        _cTypesEP = (ContentTypeExtensionPoint) manager.lookup(ContentTypeExtensionPoint.ROLE);
        _manager = manager;
    }
    
    public Map<String, ModelItem> getModelItems()
    {
        _tableRefSourceItems = new HashMap<>();
        
        ElementDefinition<String> contentTypes = FormElementDefinitionHelper.getElementDefinition(ATTRIBUTE_TABLE_REF_ID, ModelItemTypeConstants.STRING_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_CHOICE_TABLE_REF_ID", "PLUGINS_FORMS_QUESTIONS_DIALOG_CHOICE_TABLE_REF_ID_DESC", new DefaultValidator(null, true));
        contentTypes.setEnumerator(new ContentTypeEnumerator());
        
        StaticEnumerator<String> staticEnumerator = new StaticEnumerator<>();
        for (String typeId : _cTypesEP.getExtensionsIds())
        {
            ContentType contentType = _cTypesEP.getExtension(typeId);
            if (_isTableRef(contentType))
            {
                staticEnumerator.add(contentType.getLabel(), contentType.getId());
            }
        }
        contentTypes.setEnumerator(staticEnumerator);
        _tableRefSourceItems.put(contentTypes.getName(), contentTypes);
       
        return _tableRefSourceItems;
    }

    private boolean _isTableRef (ContentType cType)
    {
        return cType.isReferenceTable() && !cType.isAbstract() && !cType.isMixin();
    }
    
    public List<ViewItem> getViewItems()
    {
        List<ViewItem> viewElements = new ArrayList<>();
        
        ViewElement grid = new ViewElement();
        grid.setDefinition((ElementDefinition< ? >) _tableRefSourceItems.get(ATTRIBUTE_TABLE_REF_ID));
        viewElements.add(grid);
        
        return viewElements;
    }
    
    @Override
    public List<String> getFieldToDisableIfFormPublished()
    {
        List<String> fieldNames =  super.getFieldToDisableIfFormPublished();
        fieldNames.add(ATTRIBUTE_TABLE_REF_ID);
        return fieldNames;
    }

    public boolean remoteData()
    {
        return true;
    }

    public I18nizableText getEntry(ChoiceOption value, Map<String, Object> contextParams) throws Exception
    {
        ContentValue contentValue = (ContentValue) value.getValue();
        try 
        {
            String lang = (String) contextParams.get(LazyEnumerator.LANG_PARAMETER_KEY);
            Locale locale = lang != null ? new Locale(lang) : null;
            ModifiableContent content = contentValue.getContent();
            return new I18nizableText(content.getTitle(locale));
        }
        catch (UnknownAmetysObjectException e)
        {
            getLogger().warn("An error occured while getting label for reference table entry with id '" + contentValue.getContentId() + "' for question " + ((FormQuestion) contextParams.get("question")).getName() + " : {}", e.getMessage());
            return new I18nizableText("plugin.forms", "PLUGINS_FORMS_ENTRY_UNKNOWN_TABLE_REF_CONTENT");
        }
    }

    public Map<ChoiceOption, I18nizableText> getTypedEntries(Map<String, Object> contextParams) throws Exception
    {
        throw new UnsupportedOperationException("Method getTypedEntries can't be called for TableRefSourceType");
    }

    public Map<ChoiceOption, I18nizableText> searchEntries(Map<String, Object> contextParams, int limit, Object paginationData) throws Exception
    {
        Map<ChoiceOption, I18nizableText> entries = new LinkedHashMap<>();
        
        FormQuestion question = _getQuestionFromParam(contextParams);
        String tableRefId = question.getValue(ATTRIBUTE_TABLE_REF_ID);
        
        int offset = paginationData != null ? ((int) paginationData - 1) * limit : 0;
        
        String pattern = (String) contextParams.get(LazyEnumerator.PATTERN_PARAMETER_KEY);
        String lang = (String) contextParams.get(LazyEnumerator.LANG_PARAMETER_KEY);
        Boolean needSort = (Boolean) contextParams.getOrDefault(LazyEnumerator.SORT_PARAMETER_KEY, false);
        
        String modelId = "reference-table-search-ui." + tableRefId;
        if (_searchUIModelExtensionPoint == null)
        {
            _searchUIModelExtensionPoint = (SearchUIModelExtensionPoint) _manager.lookup(SearchUIModelExtensionPoint.ROLE); //this is a lazy loading because there is an infinite loop in lookups
        }
        SearchUIModel model = _searchUIModelExtensionPoint.getExtension(modelId);
        
        Map<String, Object> values = new HashMap<>();
        if (StringUtils.isNotBlank(pattern))
        {
            values.put("metadata-title-search", "*" + pattern + "*");
        }
        Map<String, Object> param = new HashMap<>();
        if (StringUtils.isNotBlank(lang))
        {
            param.put("language", lang);
        }
        
        Sort sort = needSort ? new Sort(SolrFieldNames.TITLE, Order.ASC) : null;
        AmetysObjectIterable<Content> results = _searcherFactory.create(model)
                .withSearchMode("simple")
                .withSort(sort != null ? List.of(sort) : List.of())
                .withLimits(offset, limit)
                .search(values, param);
        
        Locale locale = lang != null ? new Locale(lang) : null;
        for (Content content : results)
        {
            ChoiceOption choice = new ChoiceOption(content.getId());
            entries.put(choice, new I18nizableText(content.getTitle(locale)));
        }

        return entries;
    }
    
    @Override
    public String getStorageType(FormQuestion question)
    {
        return org.ametys.cms.data.type.ModelItemTypeConstants.CONTENT_ELEMENT_TYPE_ID;
    }
    
    @Override
    public String getJSRenderer()
    {
        return "Ametys.plugins.forms.helper.SearchEntriesGridHelper.renderContentChoiceList";
    }
    
    @Override
    public String getJSConverter()
    {
        return "Ametys.plugins.forms.helper.SearchEntriesGridHelper.convertContent";
    }
    
    @Override
    public Object removeEmptyOrOtherValue(Object value)
    {
        if (value == null)
        {
            return null;
        }
        
        if (value.getClass().isArray())
        {
            List<ContentValue> newVal = new ArrayList<>();
            for (ContentValue val : (ContentValue[]) value)
            {
                if (StringUtils.isNotBlank(val.getContentId()) && !val.getContentId().equals(ChoicesListQuestionType.OTHER_OPTION_VALUE))
                {
                    newVal.add(val);
                }
            }
            
            if (newVal.isEmpty())
            {
                return null;
            }
            
            return newVal.toArray(new ContentValue[newVal.size()]);
        }
        else if (StringUtils.isNotBlank(((ContentValue) value).getContentId()) && !((ContentValue) value).getContentId().equals(ChoicesListQuestionType.OTHER_OPTION_VALUE))
        {
            return value;
        }
        
        return null;
    }
    
    @Override
    public Object valueToJSONForClient(Object value, FormQuestion question, FormEntry entry, ModelItem modelItem) throws Exception
    {
        return _getComputedComplexValue(value, question, entry);
    }
    
    @Override
    public String value2String(Object value)
    {
        if (value instanceof ContentValue content)
        {
            return content.getContentId();
        }
        
        return super.value2String(value);
    }
}
