/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.question.types;

import java.util.List;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;

import org.ametys.plugins.forms.repository.Form;
import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.plugins.workflow.support.WorkflowHelper;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.SimpleViewItemGroup;
import org.ametys.runtime.model.ViewElement;
import org.ametys.runtime.model.ViewItemGroup;

import com.opensymphony.workflow.loader.WorkflowDescriptor;

/**
 * Default abstract class to represent a question type.
 * This default implementation allow to define an illustration, a description and mark the question as mandatory
 */
public abstract class AbstractFormQuestionType extends AbstractStaticFormQuestionType implements IllustrableFormQuestionType, DescriptibleFormQuestionType, MandatoryAwareFormQuestionType, ConfidentialAwareQuestionType, RestrictiveQuestionType
{
    /** The workflow helper component */
    protected WorkflowHelper _workflowHelper;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _workflowHelper = (WorkflowHelper) manager.lookup(WorkflowHelper.ROLE);
    }
    
    @Override
    protected List<ModelItem> _getModelItems()
    {
        List<ModelItem> modelItems = super._getModelItems();
        
        //DESCRIPTION
        modelItems.add(getDescriptionModelItem());

        //RESTRICTION BY WORKFLOW
        modelItems.addAll(getRestrictiveModelItems());
        
        //ILLUSTRATION & ALTERNATE TEXT
        modelItems.addAll(getIllustrationModelItems());
        
        //MANDATORY
        modelItems.add(getMandatoryModelItem());
        
        //CONFIDENTIALITY
        modelItems.add(getConfidentialityModelItem());
        
        return modelItems;
    }
    
    @Override
    protected List<ViewItemGroup> _getTabs(Form form)
    {
        if (StringUtils.isEmpty(form.getWorkflowName()))
        {
            return List.of(
                    _getMainTab(),
                    _getAdvancedTab(),
                    _getRulesTab(),
                    getIllustrationTab(getModel())
                    );
        }
        else
        {
            return List.of(
                    _getMainTab(),
                    _getAdvancedTab(),
                    _getRulesTab(),
                    getRestrictiveTab(getModel(), form),
                    getIllustrationTab(getModel())
                    );
        }
    }
    
    /**
     * Define the content of the main tab
     * @return the main tab definition
     */
    protected SimpleViewItemGroup _getMainTab() 
    {
        SimpleViewItemGroup fieldset = super._createMainTab();
        
        ViewElement title = new ViewElement();
        title.setDefinition((ElementDefinition< ? >) getModel().getModelItem(FormQuestion.ATTRIBUTE_TITLE));
        fieldset.addViewItem(title);
        
        fieldset.addViewItem(getDescriptionViewElement(getModel()));
        
        fieldset.addViewItem(getMandatoryViewElement(getModel()));
        
        return fieldset;
    }
    
    /**
     * Define the content of the advanced tab
     * @return the advanced tab definition
     */
    protected SimpleViewItemGroup _getAdvancedTab()
    {
        SimpleViewItemGroup advancedFieldset = super._createAdvancedTab();
        
        ViewElement nameForForms = new ViewElement();
        nameForForms.setDefinition((ElementDefinition< ? >) getModel().getModelItem(FormQuestion.ATTRIBUTE_NAME_FOR_FORM));
        advancedFieldset.addViewItem(nameForForms);
        
        advancedFieldset.addViewItem(getConfidentialityViewElement(getModel()));
        
        return advancedFieldset;
    }
    
    @Override
    protected ModelItem _getEntryModelItem(FormQuestion question)
    {
        // Add the mandatory validator to the entry model item
        ModelItem item = super._getEntryModelItem(question);
        ((ElementDefinition) item).setValidator(getMandatoryValidator(question));
        return item;
    }
    
    @Override
    public List<String> getFieldToDisableIfFormPublished(FormQuestion question)
    {
        List<String> fieldNames =  super.getFieldToDisableIfFormPublished(question);
        fieldNames.add(MandatoryAwareFormQuestionType.ATTRIBUTE_MANDATORY);
        fieldNames.add(ConfidentialAwareQuestionType.ATTRIBUTE_CONFIDENTIALITY);
        fieldNames.add(RestrictiveQuestionType.ATTRIBUTE_READING_CHECKBOX);
        fieldNames.add(RestrictiveQuestionType.ATTRIBUTE_READING);
        return fieldNames;
    }
    
    @Override
    public WorkflowDescriptor getWorkflowDescriptor(Form form)
    {
        return _workflowHelper.getWorkflowDescriptor(form.getWorkflowName());
    }
}
