/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.repository;

import javax.jcr.Node;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.core.group.GroupManager;
import org.ametys.plugins.forms.dao.FormDirectoryDAO;
import org.ametys.plugins.forms.dao.FormEntryDAO;
import org.ametys.plugins.repository.AmetysObjectFactory;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.data.type.ModelItemTypeExtensionPoint;
import org.ametys.plugins.repository.jcr.DefaultTraversableAmetysObjectFactory;
import org.ametys.runtime.model.DefaultElementDefinition;
import org.ametys.runtime.model.Model;
import org.ametys.runtime.model.exception.BadItemTypeException;
import org.ametys.runtime.model.exception.UnknownTypeException;
import org.ametys.runtime.model.type.ModelItemTypeConstants;

/**
 * {@link AmetysObjectFactory} for handling {@link Form}s.
 */
public class FormFactory extends DefaultTraversableAmetysObjectFactory
{
    /** JCR nodetype for form */
    public static final String FORM_NODETYPE = RepositoryConstants.NAMESPACE_PREFIX + ":form";
    
    /** Group manager */
    protected GroupManager _groupManager;
    
    /** Content type extension point */
    protected ContentTypeExtensionPoint _cTypeEP;

    /** The Form DAO */
    protected FormDirectoryDAO _formDAO;
    
    /** The form entry DAO */
    protected FormEntryDAO _formEntryDAO;
    
    /** The Form model */
    protected Model _model;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _cTypeEP = (ContentTypeExtensionPoint) manager.lookup(ContentTypeExtensionPoint.ROLE);
        _groupManager = (GroupManager) manager.lookup(GroupManager.ROLE);
        _formDAO = (FormDirectoryDAO) manager.lookup(FormDirectoryDAO.ROLE);
        _formEntryDAO = (FormEntryDAO) manager.lookup(FormEntryDAO.ROLE);
    }
    
    @Override
    public Form getAmetysObject(Node node, String parentPath) throws AmetysRepositoryException
    {
        return new Form(node, parentPath, this);
    }
    
    /**
     * Group manager getter.
     * @return The group manager
     */
    GroupManager _getGroupManager()
    {
        return _groupManager;
    }
    
    /**
     * Getter for the content type extension point
     * @return The content type EP.
     */
    ContentTypeExtensionPoint _getContentTypeExtensionPoint()
    {
        return _cTypeEP;
    }
    
    /**
     * Getter for the form directories DAO
     * @return the form directories DAO
     */
    FormDirectoryDAO getFormDirectoriesDAO()
    {
        return _formDAO;
    }
    
    /**
     * Getter for the form entry DAO
     * @return the form entry DAO
     */
    FormEntryDAO getFormEntryDAO()
    {
        return _formEntryDAO;
    }

    /**
     * Retrieves the Form model
     * @return the Form model
     */
    public Model getModel()
    {
        if (_model == null)
        {
            _model = _createFormModel();
        }
        return _model;
    }
    
    /**
     * Creates the Form model
     * @return the created model
     * @throws AmetysRepositoryException if an error occurs.
     */
    protected Model _createFormModel() throws AmetysRepositoryException
    {
        try
        {
            String role = ModelItemTypeExtensionPoint.ROLE_MODEL_AWARE_BASIC;
            return Model.of(Form.class.getName(), Form.class.getName(),
                    DefaultElementDefinition.of(Form.TITLE, false, ModelItemTypeConstants.STRING_TYPE_ID, role),
                    DefaultElementDefinition.of(Form.DESCRIPTION, false, ModelItemTypeConstants.STRING_TYPE_ID, role),
                    DefaultElementDefinition.of(Form.CREATIONDATE, false, ModelItemTypeConstants.DATETIME_TYPE_ID, role),
                    DefaultElementDefinition.of(Form.LASTMODIFICATIONDATE, false, ModelItemTypeConstants.DATETIME_TYPE_ID, role),
                    DefaultElementDefinition.of(Form.AUTHOR, false, org.ametys.cms.data.type.ModelItemTypeConstants.USER_ELEMENT_TYPE_ID, role),
                    DefaultElementDefinition.of(Form.CONTRIBUTOR, false, org.ametys.cms.data.type.ModelItemTypeConstants.USER_ELEMENT_TYPE_ID, role),
                    DefaultElementDefinition.of(Form.LIMIT_TO_ONE_ENTRY_BY_USER, false, ModelItemTypeConstants.BOOLEAN_TYPE_ID, role),
                    DefaultElementDefinition.of(Form.LIMIT_ENTRIES_ENABLED, false, ModelItemTypeConstants.BOOLEAN_TYPE_ID, role),
                    DefaultElementDefinition.of(Form.QUEUE_ENABLED, false, ModelItemTypeConstants.BOOLEAN_TYPE_ID, role),
                    DefaultElementDefinition.of(Form.MAX_ENTRIES, false, ModelItemTypeConstants.LONG_TYPE_ID, role),
                    DefaultElementDefinition.of(Form.CLOSED_MSG, false, ModelItemTypeConstants.STRING_TYPE_ID, role),
                    DefaultElementDefinition.of(Form.REMAINING_MSG, false, ModelItemTypeConstants.STRING_TYPE_ID, role),
                    DefaultElementDefinition.of(Form.QUEUE_SIZE, false, ModelItemTypeConstants.LONG_TYPE_ID, role),
                    DefaultElementDefinition.of(Form.QUEUE_CLOSED_MSG, false, ModelItemTypeConstants.STRING_TYPE_ID, role),
                    DefaultElementDefinition.of(Form.QUEUE_SENDER, false, ModelItemTypeConstants.STRING_TYPE_ID, role),
                    DefaultElementDefinition.of(Form.QUEUE_RECEIVER, false, ModelItemTypeConstants.STRING_TYPE_ID, role),
                    DefaultElementDefinition.of(Form.QUEUE_SUBJECT, false, ModelItemTypeConstants.STRING_TYPE_ID, role),
                    DefaultElementDefinition.of(Form.QUEUE_BODY, false, ModelItemTypeConstants.STRING_TYPE_ID, role),
                    DefaultElementDefinition.of(Form.ADMIN_EMAILS, true, ModelItemTypeConstants.STRING_TYPE_ID, role),
                    DefaultElementDefinition.of(Form.ADMIN_EMAILS_OTHER, false, ModelItemTypeConstants.STRING_TYPE_ID, role),
                    DefaultElementDefinition.of(Form.ADMIN_EMAIL_SUBJECT, false, ModelItemTypeConstants.STRING_TYPE_ID, role),
                    DefaultElementDefinition.of(Form.ADMIN_EMAIL_BODY, false, ModelItemTypeConstants.STRING_TYPE_ID, role),
                    DefaultElementDefinition.of(Form.RECEIPT_SENDER, false, ModelItemTypeConstants.STRING_TYPE_ID, role),
                    DefaultElementDefinition.of(Form.RECEIPT_RECEIVER, false, ModelItemTypeConstants.STRING_TYPE_ID, role),
                    DefaultElementDefinition.of(Form.RECEIPT_SUBJECT, false, ModelItemTypeConstants.STRING_TYPE_ID, role),
                    DefaultElementDefinition.of(Form.RECEIPT_BODY, false, ModelItemTypeConstants.STRING_TYPE_ID, role),
                    DefaultElementDefinition.of(Form.WORKFLOWNAME, false, ModelItemTypeConstants.STRING_TYPE_ID, role),
                    DefaultElementDefinition.of(Form.START_DATE, false, ModelItemTypeConstants.DATE_TYPE_ID, role),
                    DefaultElementDefinition.of(Form.END_DATE, false, ModelItemTypeConstants.DATE_TYPE_ID, role));
        }
        catch (UnknownTypeException | BadItemTypeException | ServiceException e)
        {
            throw new AmetysRepositoryException("An error occurred while creating the Form model", e);
        }
    }
}
