/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.forms.repository;

import java.util.List;
import java.util.stream.Collectors;

import javax.jcr.Node;
import javax.jcr.PathNotFoundException;
import javax.jcr.RepositoryException;

import org.ametys.plugins.forms.repository.FormPageRule.PageRuleType;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.CopiableAmetysObject;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.jcr.DefaultTraversableAmetysObject;

/**
 * {@link AmetysObject} for storing form
 */
public class FormPage extends DefaultTraversableAmetysObject<FormPageFactory> implements CopiableAmetysObject
{
    /** Constants for title property. */
    private static final String __PROPERTY_TITLE = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":title";
    
    /** Constant for node's name */
    private static final String __PROPERTY_RULE = "ametys-internal:rule";
    
    /**
     * Creates a {@link FormPage}.
     * @param node the node backing this {@link AmetysObject}.
     * @param parentPath the parent path in the Ametys hierarchy.
     * @param factory the {@link FormFactory} which creates the AmetysObject.
     */
    public FormPage(Node node, String parentPath, FormPageFactory factory)
    {
        super(node, parentPath, factory);
    }

    /**
     * Retrieves the title.
     * @return the title.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public String getTitle() throws AmetysRepositoryException
    {
        try
        {
            return getNode().getProperty(__PROPERTY_TITLE).getString();
        }
        catch (PathNotFoundException e)
        {
            return null;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get title property", e);
        }
    }
    
    /**
     * Get the Form to which this page belongs.
     * @return the Form to which this page belongs.
     * @throws AmetysRepositoryException if an error occurs when retrieving the form of a page
     */
    public Form getForm() throws AmetysRepositoryException
    {
        return getParent();
    }
    
    /**
     * Set the title.
     * @param title the title.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void setTitle(String title) throws AmetysRepositoryException
    {
        try
        {
            getNode().setProperty(__PROPERTY_TITLE, title);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set title property", e);
        }
    }
    
    /**
     * Get the page's questions.
     * @return the page's questions.
     * @throws AmetysRepositoryException if an error occurs when retrieving all the questions of a form
     */
    public List<FormQuestion> getQuestions() throws AmetysRepositoryException
    {
        return getChildren().stream()
                .filter(FormQuestion.class::isInstance)
                .map(FormQuestion.class::cast)
                .collect(Collectors.toList());
    }
    
    /**
     * Determines if the page has a rule
     * @return true if the page has a rule
     */
    public boolean hasRule () 
    {
        return hasChild(__PROPERTY_RULE);
    }
    
    /**
     * Get the rule
     * @return the rule
     * @throws AmetysRepositoryException if an error occurs.
     */
    public FormPageRule getRule () throws AmetysRepositoryException
    {
        return hasRule() ? getChild(__PROPERTY_RULE) : null;
    }
    
    /**
     * Set the rule for branching
     * @param ruleType the rule type
     * @param page the page to jump or skip. Can be null.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void setRule (PageRuleType ruleType, String page) throws AmetysRepositoryException
    {
        if (!hasChild(__PROPERTY_RULE))
        {
            createChild(__PROPERTY_RULE, "ametys:form-page-rule");
        }
        
        FormPageRule rule = getRule();
        
        rule.setType(ruleType);
        if (ruleType == PageRuleType.JUMP || ruleType == PageRuleType.SKIP)
        {
            rule.setPageId(page);
        }
    }
    
    /**
     * Delete the rule
     */
    public void deleteRule ()
    {
        if (hasChild(__PROPERTY_RULE))
        {
            ((FormPageRule) getChild(__PROPERTY_RULE)).remove();
        }
    }
    
    @Override
    public FormPage copyTo(ModifiableTraversableAmetysObject parent, String name) throws AmetysRepositoryException
    {
        FormPage page = parent.createChild(name, "ametys:form-page");
        page.setTitle(getTitle());
        
        Form parentForm = page.getForm();
        for (FormQuestion question : getQuestions())
        {
            String uniqueQuestionName = parentForm.findUniqueQuestionName(question.getName());
            FormQuestion questionCopy = page.createChild(uniqueQuestionName, "ametys:form-question");
            question.copyTo(questionCopy);
            questionCopy.setNameForForm(uniqueQuestionName);
        }
        page.saveChanges();
        
        return page;
    }

    @Override
    public AmetysObject copyTo(ModifiableTraversableAmetysObject parent, String name, List<String> restrictTo) throws AmetysRepositoryException
    {
        return copyTo(parent, name);
    }
}
