/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.repository.type;

import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.stream.Collectors;

import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.core.model.type.AbstractElementType;
import org.ametys.runtime.model.exception.BadItemTypeException;
import org.ametys.runtime.model.type.DataContext;

/**
 * Abstract class for matrix element types
 */
public abstract class AbstractMatrixElementType extends AbstractElementType<Matrix>
{
    /** Constant for type matrix */
    public static final String MATRIX_REPOSITORY_DATA_TYPE = "matrix";
    
    @Override
    public Matrix convertValue(Object value)
    {
        if (value == null)
        {
            return null;
        }
        else if (value instanceof String)
        {
            return _string2Matrix((String) value);
        }
        else if (value instanceof Map)
        {
            @SuppressWarnings("unchecked")
            Map<String, List<String>> castMap = (Map<String, List<String>>) value;
            return new Matrix(castMap);
        }
        else
        {
            throw new BadItemTypeException("Try to convert the non " + getId() + " JSON object '" + value + "' into a " + getId());
        }
    }
    
    @Override
    public String toString(Matrix value)
    {
        return _jsonUtils.convertObjectToJson(value);
    }

    @Override
    public boolean isSimple()
    {
        return false;
    }
    
    @Override
    public boolean isCompatible(Object value)
    {
        return super.isCompatible(value)
            || value instanceof String || value instanceof String[]
            || value instanceof Matrix || value instanceof Matrix[];
    }
    
    /**
     * Map json to matrix
     * @param value the json value
     * @return the matrix
     */
    @SuppressWarnings("unchecked")
    protected Matrix _string2Matrix(String value)
    {
        Map<String, List<String>> castMap = _jsonUtils.convertJsonToMap(value)
            .entrySet()
            .stream()
            .collect(Collectors.toMap(e -> e.getKey(), e -> (List<String>) e.getValue(), (e1, e2) -> e1, LinkedHashMap::new));
        return new Matrix(castMap);
    }
    
    @Override
    protected void _singleTypedNotEnumeratedValueToSAX(ContentHandler contentHandler, String tagName, Matrix value, DataContext context, AttributesImpl attributes)
            throws SAXException
    {
        AttributesImpl localAttributes = new AttributesImpl(attributes);
        
        XMLUtils.startElement(contentHandler, tagName, localAttributes);
        
        for (Entry<String, List<String>> lineValue : value.entrySet())
        {
            AttributesImpl optAttr = new AttributesImpl();
            optAttr.addCDATAAttribute("value", lineValue.getKey());
            XMLUtils.startElement(contentHandler, "option", optAttr);
            List<String> colValues = lineValue.getValue();
            for (String colValue : colValues)
            {
                XMLUtils.createElement(contentHandler, "value", colValue);
            }
            XMLUtils.endElement(contentHandler, "option");
        }
        
        XMLUtils.endElement(contentHandler, tagName);
    }
    
    
}
