/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.schedulable;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.components.source.impl.SitemapSource;
import org.apache.cocoon.environment.Request;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.excalibur.source.SourceResolver;
import org.apache.excalibur.source.SourceUtil;
import org.quartz.JobDataMap;
import org.quartz.JobExecutionContext;

import org.ametys.cms.schedule.AbstractSendingMailSchedulable;
import org.ametys.core.schedule.Schedulable;
import org.ametys.core.schedule.progression.ContainerProgressionTracker;
import org.ametys.core.ui.mail.StandardMailBodyHelper;
import org.ametys.core.ui.mail.StandardMailBodyHelper.MailBodyBuilder;
import org.ametys.plugins.core.schedule.Scheduler;
import org.ametys.plugins.forms.repository.Form;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;
import org.ametys.runtime.util.AmetysHomeHelper;

/**
 * {@link Schedulable} for form xls export
 */
public class ExportXlsSchedulable extends AbstractSendingMailSchedulable
{
    /** The directory under ametys home data directory for form xls export */
    public static final String FORM_EXPORT_XLS_DIR_NAME = "forms/export";
    
    /** Scheduler parameter name for form id */
    public static final String PARAM_FORM_ID = "formId";
    
    /** The avalon source resolver. */
    protected SourceResolver _sourceResolver;
    
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _sourceResolver = (SourceResolver) manager.lookup(SourceResolver.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
    }

    @Override
    protected void _doExecute(JobExecutionContext context, ContainerProgressionTracker progressionTracker) throws Exception
    {
        File xslExportDirectory = new File(AmetysHomeHelper.getAmetysHomeData(), FORM_EXPORT_XLS_DIR_NAME);
        FileUtils.forceMkdir(xslExportDirectory);
        
        JobDataMap jobDataMap = context.getJobDetail().getJobDataMap();
        String formId = jobDataMap.getString(Scheduler.PARAM_VALUES_PREFIX + PARAM_FORM_ID);
        Form form = _resolver.resolveById(formId);
        
        _generateFormExportXls(xslExportDirectory, form);
    }

    @Override
    protected I18nizableText _getSuccessMailSubject(JobExecutionContext context)
    {
        return new I18nizableText("plugin.forms", "PLUGINS_FORMS_SCHEDULABLE_EXPORT_XLS_SUBJECT_SUCCESS");
    }
    
    @Override
    protected I18nizableText _getErrorMailSubject(JobExecutionContext context)
    {
        return new I18nizableText("plugin.forms", "PLUGINS_FORMS_SCHEDULABLE_EXPORT_XLS_SUBJECT_ERROR");
    }
    
    @Override
    protected boolean _isMailBodyInHTML(JobExecutionContext context) throws Exception
    {
        return true;
    }
    
    @Override
    protected String _getSuccessMailBody(JobExecutionContext context) throws IOException
    {
        JobDataMap jobDataMap = context.getJobDetail().getJobDataMap();
        String formId = jobDataMap.getString(Scheduler.PARAM_VALUES_PREFIX + PARAM_FORM_ID);
        Form form = _resolver.resolveById(formId);
        
        try
        {
            MailBodyBuilder bodyBuilder = StandardMailBodyHelper.newHTMLBody()
                .withTitle(_getSuccessMailSubject(context));
            
            String downloadLink = _getDownloadLink(form);
            
            Map<String, I18nizableTextParameter> i18nParams = Map.of("link", new I18nizableText(downloadLink), "form", new I18nizableText(form.getTitle()));
            bodyBuilder.addMessage(new I18nizableText("plugin.forms", "PLUGINS_FORMS_SCHEDULABLE_EXPORT_XLS_BODY_SUCCESS", i18nParams));
            bodyBuilder.withLink(downloadLink, new I18nizableText("plugin.forms", "PLUGINS_FORMS_SCHEDULABLE_EXPORT_XLS_BODY_DOWNLOAD_LINK"));
            
            return bodyBuilder.build();
        }
        catch (IOException e)
        {
            getLogger().error("Failed to build HTML email body for xls export result", e);
            return null;
        }
    }
    
    /**
     * Get the link to download form xls export
     * @param form the form
     * @return the download link
     * @throws IOException if failed to build the download uri
     */
    protected String _getDownloadLink(Form form) throws IOException
    {
        String downloadLink = StringUtils.removeEndIgnoreCase(Config.getInstance().getValue("cms.url"), "/index.html");
        downloadLink += "/plugins/forms/download/" + form.getName() + ".xls?formId=" + form.getId();
        return downloadLink;
    }

    @Override
    protected String _getErrorMailBody(JobExecutionContext context, Throwable throwable)
    {
        JobDataMap jobDataMap = context.getJobDetail().getJobDataMap();
        String formId = jobDataMap.getString(Scheduler.PARAM_VALUES_PREFIX + PARAM_FORM_ID);
        Form form = _resolver.resolveById(formId);
        
        try
        {
            MailBodyBuilder bodyBuilder = StandardMailBodyHelper.newHTMLBody()
                .withTitle(_getErrorMailSubject(context));
            
            Map<String, I18nizableTextParameter> i18nParams = Map.of("form", new I18nizableText(form.getTitle()));
            bodyBuilder.addMessage(new I18nizableText("plugin.forms", "PLUGINS_FORMS_SCHEDULABLE_EXPORT_XLS_BODY_ERROR", i18nParams));
            return bodyBuilder.build();
        }
        catch (IOException e)
        {
            getLogger().error("Failed to build HTML email body for xls export result", e);
            return null;
        }
    }

    /**
     * Generate the form xls export
     * @param xslExportDirectory the xls export directory
     * @param form the form
     * @throws IOException if an error occured with files
     */
    protected void _generateFormExportXls(File xslExportDirectory, Form form) throws IOException
    {
        File formDir = new File(xslExportDirectory, form.getName());
        if (!formDir.exists())
        {
            formDir.mkdir();
        }
        
        Map<String, Object> parameters = new HashMap<>();
        parameters.put("id", form.getId());
        
        Request request = ContextHelper.getRequest(_context);
        
        SitemapSource source = null;
        File pdfTmpFile = null;
        try
        {
            // Set parameters as request attributes
            for (Entry<String, Object> param : parameters.entrySet())
            {
                request.setAttribute(param.getKey(), param.getValue());
            }
            
            // Resolve the export to the appropriate pdf url.
            source = (SitemapSource) _sourceResolver.resolveURI("cocoon://_plugins/forms/forms/entries.xls?id=" + form.getId(), null, parameters);
            
            // Save the xls into a temporary file.
            String tmpFile = form.getName() + ".tmp.xls";
            pdfTmpFile = new File(formDir, tmpFile);
            
            try (OutputStream pdfTmpOs = new FileOutputStream(pdfTmpFile); InputStream sourceIs = source.getInputStream())
            {
                SourceUtil.copy(sourceIs, pdfTmpOs);
            }
            
            // If all went well until now, rename the temporary file 
            String fileName = form.getName() + ".xls";
            File xlsFile = new File(formDir, fileName);
            if (xlsFile.exists())
            {
                xlsFile.delete();
            }
            
            if (!pdfTmpFile.renameTo(xlsFile))
            {
                throw new IOException("Fail to rename " + tmpFile + " to " + fileName);
            }
        }
        finally
        {
            if (pdfTmpFile != null)
            {
                FileUtils.deleteQuietly(pdfTmpFile);
            }
            
            if (source != null)
            {
                _sourceResolver.release(source);
            }
            
            for (Entry<String, Object> param : parameters.entrySet())
            {
                request.removeAttribute(param.getKey());
            }
        }
        
    }
}
