/*
 * Copyright 2010 Anyware Services
 * 
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.content.processing;

import java.util.Map;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.repository.Content;
import org.ametys.core.right.RightManager;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.forms.content.Form;
import org.ametys.plugins.forms.content.jcr.FormPropertiesManager;
import org.ametys.plugins.forms.content.processing.ProcessFormHelper.FormInformations;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.authentication.AccessDeniedException;
import org.ametys.runtime.authentication.AuthorizationRequiredException;
import org.ametys.web.repository.content.SharedContent;
import org.ametys.web.repository.content.WebContent;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;

/**
 * Action that processes the user submitted data on a form.
 */
public class ProcessFormAction extends ServiceableAction
{
    /** Form properties manager. */
    protected FormPropertiesManager _formPropertiesManager;

    /** The ametys object resolver */
    protected AmetysObjectResolver _ametysObjectResolver;

    /** The site manager. */
    protected SiteManager _siteManager;

    /** Rights manager */
    protected RightManager _rightManager;
    
    /** current user provider */
    protected CurrentUserProvider _currentUserProvied;

    /** The process form helper */
    protected ProcessFormHelper _processFormHelper;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _processFormHelper = (ProcessFormHelper) serviceManager.lookup(ProcessFormHelper.ROLE);
        _formPropertiesManager = (FormPropertiesManager) serviceManager.lookup(FormPropertiesManager.ROLE);
        _ametysObjectResolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _siteManager = (SiteManager) serviceManager.lookup(SiteManager.ROLE);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
        _currentUserProvied = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
    }

    @Override
    public Map act(Redirector redirector, org.apache.cocoon.environment.SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Request request = ObjectModelHelper.getRequest(objectModel);

        String pluginName = (String) request.getAttribute("pluginName");
        String siteName = (String) request.getAttribute("site");
        Site site = null;

        Content content = (Content) request.getAttribute(Content.class.getName());
        if (content != null && content instanceof SharedContent)
        {
            siteName = _getSiteNameFromSharedContent ((SharedContent) content);
            site = _getSiteFromSharedContent((SharedContent) content);
        }
        else
        {
            site = _siteManager.getSite(siteName);
        }
        
        String id = request.getParameter(ProcessFormHelper.PARAM_FORM_ID);
        if (StringUtils.isEmpty(id))
        {
            throw new IllegalArgumentException("A form ID must be provided.");
        }

        Form form = _formPropertiesManager.getForm(siteName, id);
        if (form == null)
        {
            throw new IllegalArgumentException("No form definition exists for ID " + id + " and site " + siteName);
        }
        
        if (content == null)
        {
            content = _ametysObjectResolver.resolveById(form.getContentId()); 
        }

        // Protecting the form
        // The right is already checked in GetContentAction, but we cannot assume that the user will not sumbit directly here
        if (!_rightManager.currentUserHasReadAccess(content))
        {
            UserIdentity user = _currentUserProvied.getUser();
            if (user == null)
            {
                throw new AuthorizationRequiredException();
            }
            else
            {
                throw new AccessDeniedException("User '" + UserIdentity.userIdentityToString(user) + "' can not post form for content '" + content.getId() + "'");
            }
        }

        FormInformations formInfos = _processFormHelper.processForm(form, site, pluginName);
        if (formInfos.getFormErrors().hasErrors())
        {
            request.setAttribute("form-errors", formInfos.getFormErrors());
            return null;
        }
        
        if (StringUtils.isNotBlank(formInfos.getRedirection()))
        {
            redirector.globalRedirect(false, formInfos.getRedirection());
        }
        
        return EMPTY_MAP;
    }
    
    private String _getSiteNameFromSharedContent (SharedContent sharedContent)
    {
        SharedContent defaultContentContent = _ametysObjectResolver.resolveById(sharedContent.getId());
        
        Content initialContent = defaultContentContent.getInitialContent();
        if (initialContent != null && initialContent instanceof WebContent)
        {
            return ((WebContent) initialContent).getSiteName();
        }
        
        return null;
    }
    
    private Site _getSiteFromSharedContent (SharedContent sharedContent)
    {
        SharedContent defaultContentContent = _ametysObjectResolver.resolveById(sharedContent.getId());
        
        Content initialContent = defaultContentContent.getInitialContent();
        if (initialContent != null && initialContent instanceof WebContent)
        {
            return ((WebContent) initialContent).getSite();
        }
        
        return null;
    }
}
