/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.content.workflow;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.xml.xpath.PrefixResolver;
import org.apache.excalibur.xml.xpath.XPathProcessor;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import org.ametys.core.util.URIUtils;
import org.ametys.plugins.forms.content.Field;
import org.ametys.plugins.forms.content.Fieldset;
import org.ametys.plugins.forms.content.Form;
import org.ametys.plugins.forms.content.Field.FieldType;

/**
 * Form parser.
 */
public class FormParser
{

    /** The rich text prefix resolver. */
    public static final PrefixResolver RICH_TEXT_PREFIX_RESOLVER = new RichTextPrefixResolver();
    
    /** The label XML element name. */
    protected static final String _LABEL_ELEMENT = "label";
    
    /** The input XML element name. */
    protected static final String _INPUT_ELEMENT = "input";
    
    /** The textarea XML element name. */
    protected static final String _TEXTAREA_ELEMENT = "textarea";
    
    /** The select XML element name. */
    protected static final String _SELECT_ELEMENT = "select";
    
    /** The option XML element name. */
    protected static final String _OPTION_ELEMENT = "option";
    
    /** The captcha XML element name. */
    protected static final String _CAPTCHA_ELEMENT = "captcha";
    
    /** The fieldset XML element name. */
    protected static final String _FIELDSET_ELEMENT = "fieldset";
    
    /** The fieldset XML element name. */
    protected static final String _LEGEND_ELEMENT = "legend";
    
    /** An XPath processor. */
    protected XPathProcessor _xpathProcessor;
    
    /**
     * Default constructor.
     * @param xpathProcessor the xpath processor
     */
    public FormParser(XPathProcessor xpathProcessor)
    {
        _xpathProcessor = xpathProcessor;
    }

    /**
     * Return a list of nodes selected by an xpath expression.
     * @param context the context node.
     * @param xpathExpression the expression used to select nodes.
     * @return a list of nodes selected by an xpath expression.
     */
    public List<Node> getNodesAsList(Node context, String xpathExpression)
    {
        NodeList selectNodeList = _xpathProcessor.selectNodeList(context, xpathExpression, RICH_TEXT_PREFIX_RESOLVER);
        ArrayList<Node> toReturn = null;
        if (selectNodeList != null)
        {
            toReturn = new ArrayList<>();
            for (int i = 0; i < selectNodeList.getLength(); i++)
            {
                toReturn.add(selectNodeList.item(i));
            }
        }
        return toReturn;
    }

    /**
     * Parses the form.
     * @param formNode the node to parse (must be a node &lt;html:form type="cms"&gt;)
     * @return the extracted Form.
     */
    public Form parseForm(Node formNode)
    {
        if (!formNode.getNodeName().equals("form") && !"cms".equals(_getAttribute(formNode, "type")))
        {
            throw new IllegalArgumentException("The form node must be of type <form type='cms'>");
        }
        
        Form form = new Form();
        
        String id = StringUtils.defaultString(_getAttribute(formNode, "id"));
        String label = StringUtils.defaultString(_getAttribute(formNode, "label"));
        String receiptFieldId = StringUtils.defaultString(_getAttribute(formNode, "receipt_to"));
        String receiptFieldFromAddress = StringUtils.defaultString(_getAttribute(formNode, "receipt_from"));
        String receiptFieldSubject = StringUtils.defaultString(_getAttribute(formNode, "receipt_subject"));
        String receiptFieldBody = StringUtils.defaultString(_getAttribute(formNode, "receipt_body"));
        String redirectTo = StringUtils.defaultString(_getAttribute(formNode, "redirect"));
        String emails = StringUtils.defaultString(_getAttribute(formNode, "processing_emails"));
        String workflowName = StringUtils.defaultString(_getAttribute(formNode, "workflow"));
        String limit = StringUtils.defaultString(_getAttribute(formNode, "limit"));
        String places = StringUtils.defaultString(_getAttribute(formNode, "places"));
        String noPlaces = StringUtils.defaultString(_getAttribute(formNode, "no_places"));
        
        emails = URIUtils.decode(emails);
        
        Set<String> emailSet = new HashSet<>();
        for (String email : emails.split("[;, \n]"))
        {
            if (StringUtils.isNotBlank(email))
            {
                emailSet.add(email.trim());
            }
        }
        
        _parseFields(formNode, form);
        
        form.setId(id);
        form.setLabel(label);
        form.setReceiptFieldId(receiptFieldId);
        form.setReceiptFieldFromAddress(receiptFieldFromAddress);
        form.setReceiptFieldSubject(receiptFieldSubject);
        form.setReceiptFieldBody(receiptFieldBody);
        form.setNotificationEmails(emailSet);
        form.setRedirectTo(redirectTo);
        form.setWorkflowName(workflowName);
        form.setLimit(limit);
        form.setRemainingPlaces(places);
        form.setNoRemainingPlaces(noPlaces);
        
        return form;
    }
    
    /**
     * Return the value of an attribute of an input
     * 
     * @param node the node
     * @param attributeName the name of the attribute
     * @return the value of an attribute of an input
     */
    public String _getAttribute(Node node, String attributeName)
    {
        NamedNodeMap attributes = node.getAttributes();
        Node idAttribute = attributes.getNamedItem(attributeName);
        String toReturn;
        if (idAttribute != null)
        {
            toReturn = idAttribute.getNodeValue();
        }
        else
        {
            toReturn = null;
        }
        return toReturn;
    }

    /**
     * Parse the fields of a form.
     * @param formNode the form node.
     * @param form the form object.
     */
    protected void _parseFields(Node formNode, Form form)
    {
        List<Field> fields = new ArrayList<>();
        List<Fieldset> fieldsets = new ArrayList<>();
        
        Map<String, String> labels = new HashMap<>();
        
        NodeList nodeList = _xpathProcessor.selectNodeList(formNode, "descendant::html:*", RICH_TEXT_PREFIX_RESOLVER);
        for (int i = 0; i < nodeList.getLength(); i++)
        {
            Node node = nodeList.item(i);
            String localName = node.getLocalName();
            
            if (_LABEL_ELEMENT.equals(localName))
            {
                _processLabel(node, labels);
            }
            else if (_INPUT_ELEMENT.equals(localName))
            {
                Field field = _processInput(node);
                if (field != null)
                {
                    fields.add(field);
                }
            }
            else if (_TEXTAREA_ELEMENT.equals(localName))
            {
                Field field = _processTextarea(node);
                if (field != null)
                {
                    fields.add(field);
                }
            }
            else if (_SELECT_ELEMENT.equals(localName))
            {
                Field field = _processSelect(node);
                if (field != null)
                {
                    fields.add(field);
                }
            }
            else if (_CAPTCHA_ELEMENT.equals(localName))
            {
                Field field = _processCaptcha(node);
                if (field != null)
                {
                    fields.add(field);
                }
            }
            else if (_FIELDSET_ELEMENT.equals(localName))
            {
                Fieldset fieldset = _processFieldset(node);
                if (fieldset != null)
                {
                    fieldsets.add(fieldset);
                }
            }
        }
        
        // Fill in the field labels with the extracted labels.
        for (Field field : fields)
        {
            if (labels.containsKey(field.getId()))
            {
                field.setLabel(labels.get(field.getId()));
            }
        }
        
        form.setFields(fields);
        form.setFieldsets(fieldsets);
    }
    
    /**
     * Process a fieldset.
     * @param fieldsetNode the fielset DOM node.
     * @return the fieldset.
     */
    protected Fieldset _processFieldset(Node fieldsetNode)
    {
        Fieldset fieldset = new Fieldset();
        
        Node legendNode = fieldsetNode.getFirstChild();
        if (_LEGEND_ELEMENT.equals(legendNode.getLocalName()))
        {
            fieldset.setLabel(legendNode.getTextContent());
        }
        
        List<Node> fieldNodes = getNodesAsList(fieldsetNode, "descendant::html:*");
        for (Node fieldNode : fieldNodes)
        {
            String id = _getAttribute(fieldNode, "id");
            if (StringUtils.isNotEmpty(id))
            {
                fieldset.getFieldIds().add(id);
            }
        }
        
        return fieldset;
    }

    /**
     * Process a label.
     * @param labelNode the label DOM node.
     * @param labels the label map.
     */
    protected void _processLabel(Node labelNode, Map<String, String> labels)
    {
        String id = _getAttribute(labelNode, "for");
        String label = labelNode.getTextContent().trim();
        // Trim the ending colon if present.
        if (label.endsWith(":"))
        {
            label = label.substring(0, label.length() - 1).trim();
        }
        
        if (StringUtils.isNotEmpty(id))
        {
            labels.put(id, label);
        }
    }
    
    /**
     * Process an input.
     * @param inputNode the input DOM node.
     * @return the input as a Field.
     */
    protected Field _processInput(Node inputNode)
    {
        Field field = null;
        
        FieldType type = _getInputType(inputNode);
        
        if (type != null)
        {
            field = new Field(type);
            
            _processAttributes(inputNode, field);
        }
        
        return field;
    }

    /**
     * Process an textarea.
     * @param textareaNode the textarea DOM node.
     * @return the textarea as a Field.
     */
    protected Field _processTextarea(Node textareaNode)
    {
        Field field = new Field(FieldType.TEXTAREA);
        
        _processAttributes(textareaNode, field);
        
        return field;
    }

    /**
     * Process a select.
     * @param selectNode the select DOM node.
     * @return the select as a Field.
     */
    protected Field _processSelect(Node selectNode)
    {
        Field field = new Field(FieldType.SELECT);
        
        _processAttributes(selectNode, field);
        
        Map<String, String> optionProperties = new HashMap<>();
        
        List<Node> optionNodes = getNodesAsList(selectNode, "html:" + _OPTION_ELEMENT);
        
        int i = 0;
        for (Node optionNode : optionNodes)
        {
            String value = _getAttribute(optionNode, "value");
            String cost = _getAttribute(optionNode, "cost");
            String label = optionNode.getTextContent();
            
            optionProperties.put("option-" + i + "-value", value);
            optionProperties.put("option-" + i + "-label", label);
            optionProperties.put("option-" + i + "-cost", cost);
            i++;
        }
        
        field.getProperties().putAll(optionProperties);
        
        return field;
    }
    
    /**
     * Process a captcha.
     * @param captchaNode the captcha DOM node.
     * @return the captcha as a Field.
     */
    protected Field _processCaptcha(Node captchaNode)
    {
        Field field = new Field(FieldType.CAPTCHA);
        
        _processAttributes(captchaNode, field);
        
        return field;
    }
    
    /**
     * Get the input type.
     * @param inputNode the input node.
     * @return the field type.
     */
    protected FieldType _getInputType(Node inputNode)
    {
        FieldType type = null;
        
        String typeAttr = _getAttribute(inputNode, "type");
        if ("text".equals(typeAttr))
        {
            type = FieldType.TEXT;
        }
        else if ("hidden".equals(typeAttr))
        {
            type = FieldType.HIDDEN;
        }
        else if ("password".equals(typeAttr))
        {
            type = FieldType.PASSWORD;
        }
        else if ("checkbox".equals(typeAttr))
        {
            type = FieldType.CHECKBOX;
        }
        else if ("radio".equals(typeAttr))
        {
            type = FieldType.RADIO;
        }
        else if ("file".equals(typeAttr))
        {
            type = FieldType.FILE;
        }
        else if ("cost".equals(typeAttr))
        {
            type = FieldType.COST;
        }
        
        return type;
    }
    
    /**
     * Process common field attributes and properties.
     * @param node the field node.
     * @param field the field object to fill.
     */
    protected void _processAttributes(Node node, Field field)
    {
        String id = _getAttribute(node, "id");
        String name = _getAttribute(node, "name");
        
        Map<String, String> properties = new HashMap<>();
        NamedNodeMap atts = node.getAttributes();
        for (int i = 0; i < atts.getLength(); i++)
        {
            Node attribute = atts.item(i);
            String attrName = attribute.getLocalName();
            if (!"id".equals(attrName) && !"name".equals(attrName) && !"type".equals(attrName))
            {
                String value = attribute.getNodeValue();
                properties.put(attrName, value);
            }
        }
        
        field.setId(id);
        field.setName(name);
        field.getProperties().putAll(properties);
    }
    
    /**
     * XML prefix resolver which declares docbook and html namespaces.
     */
    public static class RichTextPrefixResolver implements PrefixResolver
    {
        /**
         * The declared XML namespaces.
         */
        public static final Map<String, String> NAMESPACES = new HashMap<>();
        
        static
        {
            NAMESPACES.put("docbook", "http://docbook.org/ns/docbook");
            NAMESPACES.put("html", "http://www.w3.org/1999/xhtml");
            NAMESPACES.put("xlink", "http://www.w3.org/1999/xlink");
        }
        
        public String prefixToNamespace(String prefix)
        {
            return NAMESPACES.get(prefix);
        }
    }
}
