/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.dao;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import javax.jcr.RepositoryException;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.ProcessingException;
import org.apache.commons.lang3.StringUtils;
import org.apache.jackrabbit.JcrConstants;

import org.ametys.cms.search.model.SystemPropertyExtensionPoint;
import org.ametys.cms.search.ui.model.SearchUIColumn;
import org.ametys.cms.search.ui.model.SearchUIColumnHelper;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.ObservationManager;
import org.ametys.core.right.RightManager;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.plugins.forms.FormEvents;
import org.ametys.plugins.forms.actions.GetFormEntriesAction;
import org.ametys.plugins.forms.helper.FormElementDefinitionHelper;
import org.ametys.plugins.forms.helper.LimitedEntriesHelper;
import org.ametys.plugins.forms.question.FormQuestionType;
import org.ametys.plugins.forms.question.types.MultipleAwareQuestionType;
import org.ametys.plugins.forms.question.types.RestrictiveAwareQuestionType;
import org.ametys.plugins.forms.question.types.impl.ChoicesListQuestionType;
import org.ametys.plugins.forms.question.types.impl.FileQuestionType;
import org.ametys.plugins.forms.question.types.impl.MatrixQuestionType;
import org.ametys.plugins.forms.question.types.impl.RichTextQuestionType;
import org.ametys.plugins.forms.repository.Form;
import org.ametys.plugins.forms.repository.FormEntry;
import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.plugins.forms.rights.FormsDirectoryRightAssignmentContext;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.BooleanExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.workflow.support.WorkflowProvider;
import org.ametys.plugins.workflow.support.WorkflowProvider.AmetysObjectWorkflow;
import org.ametys.runtime.authentication.AccessDeniedException;
import org.ametys.runtime.authentication.AuthorizationRequiredException;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.DefinitionContext;
import org.ametys.runtime.model.Model;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.type.ModelItemTypeConstants;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.parameters.ParametersManager;

import com.opensymphony.workflow.spi.Step;

/**
 * Form entry DAO.
 */
public class FormEntryDAO extends AbstractLogEnabled implements Serviceable, Component
{
    /** The Avalon role name. */
    public static final String ROLE = FormEntryDAO.class.getName();

    /** Name for entries root jcr node */
    public static final String ENTRIES_ROOT = "ametys-internal:form-entries";
    
    /** The right id to consult form entries */
    public static final String HANDLE_FORMS_ENTRIES_RIGHT_ID = "Form_Entries_Rights_Data";
    
    /** The right id to delete form entries */
    public static final String DELETE_FORMS_ENTRIES_RIGHT_ID = "Runtime_Rights_Forms_Entry_Delete";
    
    /** Ametys object resolver. */
    protected AmetysObjectResolver _resolver;
    /** The parameters manager */
    protected ParametersManager _parametersManager;
    /** Observer manager. */
    protected ObservationManager _observationManager;
    /** The current user provider. */
    protected CurrentUserProvider _currentUserProvider;
    /** The handling limited entries helper */
    protected LimitedEntriesHelper _handleLimitedEntriesHelper;
    /** The rights manager */
    protected RightManager _rightManager;
    /** The current user provider */
    protected WorkflowProvider _workflowProvider;
    /** The user manager */
    protected UserManager _userManager;
    /** The system property extension point */
    protected SystemPropertyExtensionPoint _systemPropertyEP;
    /** The form element definition helper */
    protected FormElementDefinitionHelper _formElementDefinitionHelper;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _parametersManager = (ParametersManager) serviceManager.lookup(ParametersManager.ROLE);
        _observationManager = (ObservationManager) serviceManager.lookup(ObservationManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) serviceManager.lookup(CurrentUserProvider.ROLE);
        _handleLimitedEntriesHelper = (LimitedEntriesHelper) serviceManager.lookup(LimitedEntriesHelper.ROLE);
        _rightManager = (RightManager) serviceManager.lookup(RightManager.ROLE);
        _workflowProvider = (WorkflowProvider) serviceManager.lookup(WorkflowProvider.ROLE);
        _userManager = (UserManager) serviceManager.lookup(UserManager.ROLE);
        _systemPropertyEP = (SystemPropertyExtensionPoint) serviceManager.lookup(SystemPropertyExtensionPoint.ROLE);
        _formElementDefinitionHelper = (FormElementDefinitionHelper) serviceManager.lookup(FormElementDefinitionHelper.ROLE);
    }
    
    /**
     * Check if a user have handle data right on a form element as ametys object
     * @param userIdentity the user
     * @param formElement the form element
     * @return true if the user handle data right for a form element
     */
    public boolean hasHandleDataRightOnForm(UserIdentity userIdentity, AmetysObject formElement)
    {
        return _rightManager.hasRight(userIdentity, HANDLE_FORMS_ENTRIES_RIGHT_ID, formElement) == RightResult.RIGHT_ALLOW;
    }
    
    /**
     * Check handle data right for a form element as ametys object
     * @param formElement the form element as ametys object
     */
    public void checkHandleDataRight(AmetysObject formElement)
    {
        UserIdentity user = _currentUserProvider.getUser();
        if (user == null)
        {
            // User not yet authenticated
            throw new AuthorizationRequiredException();
        }
        
        if (!hasHandleDataRightOnForm(user, formElement))
        {
            throw new AccessDeniedException("User '" + user + "' tried to handle form data without convenient right [" + HANDLE_FORMS_ENTRIES_RIGHT_ID + "]");
        }
    }
    
    /**
     * Gets properties of a form entry
     * @param id The id of the form entry
     * @return The properties
     */
    @Callable (rights = Callable.NO_CHECK_REQUIRED)
    public Map<String, Object> getFormEntryProperties (String id)
    {
        // Assume that no read access is checked (required for bus message target)
        try
        {
            FormEntry entry = _resolver.resolveById(id);
            return getFormEntryProperties(entry);
        }
        catch (UnknownAmetysObjectException e)
        {
            getLogger().warn("Can't find entry with id: {}. It probably has just been deleted", id, e);
            Map<String, Object> infos = new HashMap<>();
            infos.put("id", id);
            return infos;
        }
    }
    
    /**
     * Gets properties of a form entry
     * @param entry The form entry
     * @return The properties
     */
    public Map<String, Object> getFormEntryProperties (FormEntry entry)
    {
        Map<String, Object> properties = new HashMap<>();
        
        properties.put("id", entry.getId());
        properties.put("formId", entry.getForm().getId());
        properties.put("rights", _getUserRights(entry));
        
        return properties;
    }
    
    /**
     * Get user rights for the given form entry
     * @param entry the form entry
     * @return the set of rights
     */
    protected Set<String> _getUserRights (FormEntry entry)
    {
        UserIdentity user = _currentUserProvider.getUser();
        return _rightManager.getUserRights(user, entry);
    }
    
    /**
     * Creates a {@link FormEntry}.
     * @param form The parent form
     * @return return the form entry
     */
    public FormEntry createEntry(Form form)
    {
        ModifiableTraversableAmetysObject entriesRoot;
        if (form.hasChild(ENTRIES_ROOT))
        {
            entriesRoot = form.getChild(ENTRIES_ROOT);
        }
        else
        {
            entriesRoot = form.createChild(ENTRIES_ROOT, "ametys:collection");
        }
        // Find unique name
        String originalName = "entry";
        String uniqueName = originalName + "-1";
        int index = 2;
        while (entriesRoot.hasChild(uniqueName))
        {
            uniqueName = originalName + "-" + (index++);
        }
        FormEntry entry =  (FormEntry) entriesRoot.createChild(uniqueName, "ametys:form-entry");
        
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put("form", form);
        _observationManager.notify(new Event(FormEvents.FORM_MODIFIED, _currentUserProvider.getUser(), eventParams));
        
        return entry;
    }
    
    /**
     * Get the search model configuration to search form entries
     * @param formId the identifier of form
     * @return The search model configuration
     * @throws ProcessingException If an error occurred
     */
    @Callable (rights = HANDLE_FORMS_ENTRIES_RIGHT_ID, rightContext = FormsDirectoryRightAssignmentContext.ID, paramIndex = 0)
    public Map<String, Object> getSearchModelConfiguration (String formId) throws ProcessingException
    {
        Map<String, Object> result = new HashMap<>();
        
        Form form = _resolver.resolveById(formId);
        result.put("criteria", _getCriteria(form));
        result.put("columns", _getColumns(form));

        result.put("searchUrlPlugin", "forms");
        result.put("searchUrl", "form/entries.json");
        result.put("pageSize", 50);
        return result;
    }
    
    /**
     * Get criteria to search form entries
     * @param form the form
     * @return the criteria as JSON
     */
    protected Map<String, Object> _getCriteria(Form form)
    {
        // Currently, return no criteria for search entries tool
        return Map.of();
    }
    
    /**
     * Get the columns for search form entries
     * @param form the form
     * @return the columns as JSON
     * @throws ProcessingException if an error occurred
     */
    protected List<Map<String, Object>> _getColumns(Form form) throws ProcessingException
    {
        List<SearchUIColumn> columns = new ArrayList<>();
        
        Model formEntryModel = getFormEntryModel(form);
        
        ModelItem idAttribute = formEntryModel.getModelItem(FormEntry.ATTRIBUTE_ID);
        SearchUIColumn idColumn = SearchUIColumnHelper.createModelItemColumn(idAttribute);
        idColumn.setWidth(80);
        columns.add(idColumn);
        
        ModelItem userAttribute = formEntryModel.getModelItem(FormEntry.ATTRIBUTE_USER);
        SearchUIColumn userColumn = SearchUIColumnHelper.createModelItemColumn(userAttribute);
        userColumn.setWidth(150);
        columns.add(userColumn);
        
        ModelItem submitDateAttribute = formEntryModel.getModelItem(FormEntry.ATTRIBUTE_SUBMIT_DATE);
        SearchUIColumn submitDateColumn = SearchUIColumnHelper.createModelItemColumn(submitDateAttribute);
        submitDateColumn.setWidth(150);
        columns.add(submitDateColumn);
        
        for (ModelItem modelItem : formEntryModel.getModelItems())
        {
            String modelItemName = modelItem.getName();
            if (!modelItemName.equals(FormEntry.ATTRIBUTE_IP)
                 && !modelItemName.equals(FormEntry.ATTRIBUTE_ACTIVE)
                 && !modelItemName.equals(FormEntry.ATTRIBUTE_SUBMIT_DATE)
                 && !modelItemName.equals(FormEntry.ATTRIBUTE_ID)
                 && !modelItemName.equals(FormEntry.ATTRIBUTE_USER)
                 && !modelItemName.startsWith(ChoicesListQuestionType.OTHER_PREFIX_DATA_NAME))
            {
                SearchUIColumn<? extends ModelItem> column = SearchUIColumnHelper.createModelItemColumn(modelItem);
                
                FormQuestion question = form.getQuestion(modelItemName);
                if (question != null)
                {
                    FormQuestionType type = question.getType();
                    
                    column.setRenderer(Optional.ofNullable(type.getJSRenderer(question))
                                               .filter(StringUtils::isNotBlank));
                    
                    column.setConverter(Optional.ofNullable(type.getJSConverter(question))
                                                .filter(StringUtils::isNotBlank));
                    
                    column.setSortable(_isSortable(question));
                }
                
                columns.add(column);
            }
        }
        
        List<Map<String, Object>>  columnsInfo = new ArrayList<>();
        DefinitionContext definitionContext = DefinitionContext.newInstance();
        for (SearchUIColumn column : columns)
        {
            columnsInfo.add(column.toJSON(definitionContext));
        }
        
        if (form.isQueueEnabled())
        {
            columnsInfo.add(
                Map.of("name", FormEntry.SYSTEM_ATTRIBUTE_PREFIX + GetFormEntriesAction.QUEUE_STATUS,
                        "label", new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUEUE_STATUS_COLUMN_TITLE_LABEL"),
                        "type", ModelItemTypeConstants.BOOLEAN_TYPE_ID,
                        "path", FormEntry.SYSTEM_ATTRIBUTE_PREFIX + GetFormEntriesAction.QUEUE_STATUS
                 )
            );
        }
        
        columnsInfo.add(
            Map.of("name", FormEntry.SYSTEM_ATTRIBUTE_PREFIX + GetFormEntriesAction.FORM_ENTRY_ACTIVE,
                    "label", new I18nizableText("plugin.forms", "PLUGINS_FORMS_ENTRY_ACTIVE_COLUMN_TITLE_LABEL"),
                    "type", ModelItemTypeConstants.BOOLEAN_TYPE_ID,
                    "path", FormEntry.SYSTEM_ATTRIBUTE_PREFIX + GetFormEntriesAction.FORM_ENTRY_ACTIVE,
                    "hidden", true
             )
        );
        
        return columnsInfo;
    }
    
    /**
     * <code>true</code> if the column link to the question is sortable
     * @param question the question
     * @return <code>true</code> if the column link to the question is sortable
     */
    protected boolean _isSortable(FormQuestion question)
    {
        FormQuestionType type = question.getType();
        if (type instanceof MultipleAwareQuestionType multipleType && multipleType.isMultiple(question))
        {
            return false;
        }
        
        if (type instanceof MatrixQuestionType || type instanceof RichTextQuestionType || type instanceof FileQuestionType)
        {
            return false;
        }
        
        return true;
    }

    /**
     * Deletes a {@link FormEntry}.
     * @param id The id of the form entry to delete
     * @return The entry data
     */
    @Callable (rights = DELETE_FORMS_ENTRIES_RIGHT_ID, rightContext = FormsDirectoryRightAssignmentContext.ID, paramIndex = 0)
    public Map<String, String> deleteEntry (String id)
    {
        Map<String, String> result = new HashMap<>();
        
        FormEntry entry = _resolver.resolveById(id);
        
        _handleLimitedEntriesHelper.deactivateEntry(id);
        
        Form form = entry.getForm();
        entry.remove();
        
        form.saveChanges();
        
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put("form", form);
        _observationManager.notify(new Event(FormEvents.FORM_MODIFIED, _currentUserProvider.getUser(), eventParams));
        
        result.put("entryId", id);
        result.put("formId", form.getId());
        result.put("hasEntries", String.valueOf(form.hasEntries()));
        
        return result;
    }
    
    /**
     * Delete all entries of a form
     * @param id The id of the form
     * @return the deleted entries data
     */
    @Callable (rights = DELETE_FORMS_ENTRIES_RIGHT_ID, rightContext = FormsDirectoryRightAssignmentContext.ID, paramIndex = 0)
    public Map<String, Object> clearEntries(String id)
    {
        Map<String, Object> result = new HashMap<>();
        List<String> entryIds = new ArrayList<>();
        Form form = _resolver.resolveById(id);
        
        for (FormEntry entry: form.getEntries())
        {
            entryIds.add(entry.getId());
            entry.remove();
        }
        form.saveChanges();
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put("form", form);
        _observationManager.notify(new Event(FormEvents.FORM_MODIFIED, _currentUserProvider.getUser(), eventParams));
        
        result.put("ids", entryIds);
        result.put("formId", form.getId());
        
        return result;
    }

    /**
     * Retrieves the current step id of the form entry
     * @param entry The form entry
     * @return the current step id
     * @throws AmetysRepositoryException if an error occurs.
     */
    public Long getCurrentStepId(FormEntry entry) throws AmetysRepositoryException
    {
        AmetysObjectWorkflow workflow = _workflowProvider.getAmetysObjectWorkflow(entry);
        try
        {
            Step currentStep = (Step) workflow.getCurrentSteps(entry.getWorkflowId()).iterator().next();
            return Long.valueOf(currentStep.getStepId());
        }
        catch (AmetysRepositoryException e)
        {
            return RestrictiveAwareQuestionType.INITIAL_WORKFLOW_ID; // can occur when entry has just been created and workflow is not yet initialized
        }
    }
    
    /**
     * Get the form entry model
     * @param form the form
     * @return the form entry model
     */
    public Model getFormEntryModel(Form form)
    {
        List<ModelItem> items = new ArrayList<>();
        for (FormQuestion question : form.getQuestions())
        {
            FormQuestionType type = question.getType();
            if (!type.onlyForDisplay(question))
            {
                Model entryModel = question.getType().getEntryModel(question);
                for (ModelItem modelItem : entryModel.getModelItems())
                {
                    items.add(modelItem);
                }
                
                if (type instanceof ChoicesListQuestionType cLType)
                {
                    ModelItem otherFieldModel = cLType.getOtherFieldModel(question);
                    if (otherFieldModel != null)
                    {
                        items.add(otherFieldModel);
                    }
                }
            }
        }
        
        items.add(_formElementDefinitionHelper.getElementDefinition(FormEntry.ATTRIBUTE_ID, ModelItemTypeConstants.LONG_TYPE_ID, "PLUGIN_FORMS_MODEL_ITEM_ID_LABEL", null, null));
        items.add(_formElementDefinitionHelper.getElementDefinition(FormEntry.ATTRIBUTE_USER, org.ametys.cms.data.type.ModelItemTypeConstants.USER_ELEMENT_TYPE_ID, "PLUGIN_FORMS_MODEL_ITEM_USER_LABEL", null, null));
        items.add(_formElementDefinitionHelper.getElementDefinition(FormEntry.ATTRIBUTE_ACTIVE, ModelItemTypeConstants.BOOLEAN_TYPE_ID, "PLUGIN_FORMS_MODEL_ITEM_ACTIVE_LABEL", null, null));
        items.add(_formElementDefinitionHelper.getElementDefinition(FormEntry.ATTRIBUTE_SUBMIT_DATE, ModelItemTypeConstants.DATETIME_TYPE_ID, "PLUGIN_FORMS_MODEL_ITEM_SUBMISSION_DATE_LABEL", null, null));
        items.add(_formElementDefinitionHelper.getElementDefinition(FormEntry.ATTRIBUTE_IP, ModelItemTypeConstants.STRING_TYPE_ID, "PLUGIN_FORMS_MODEL_ITEM_IP_LABEL", null, null));
        items.add(_formElementDefinitionHelper.getElementDefinition(FormEntry.ATTRIBUTE_ANONYMIZATION_DATE, ModelItemTypeConstants.DATETIME_TYPE_ID, "PLUGIN_FORMS_MODEL_ITEM_ANONYMIZATION_DATE_LABEL", null, null));
        
        if (form.hasWorkflow())
        {
            items.add(_systemPropertyEP.getExtension("workflowName"));
            items.add(_systemPropertyEP.getExtension("workflowStep"));
        }
        
        return Model.of(
            "form.entry.model.id",
            "form.entry.model.family.id",
            items.toArray(ModelItem[]::new)
        );
    }
    
    /**
     * Get the form entries
     * @param form the form
     * @param onlyActiveEntries <code>true</code> to have only active entries
     * @param sorts the list of sort
     * @return the form entries
     */
    public List<FormEntry> getFormEntries(Form form, boolean onlyActiveEntries, List<Sort> sorts)
    {
        return getFormEntries(form, onlyActiveEntries, null, sorts);
    }
    
    /**
     * Get the form entries
     * @param form the form
     * @param onlyActiveEntries <code>true</code> to have only active entries
     * @param additionalEntryFilterExpr the additional entry filter expression. Can be null.
     * @param sorts the list of sort
     * @return the form entries
     */
    public List<FormEntry> getFormEntries(Form form, boolean onlyActiveEntries, Expression additionalEntryFilterExpr, List<Sort> sorts)
    {
        try
        {
            String uuid = form.getNode().getIdentifier();
            String xpathQuery = "//element(*, ametys:form)[@" + JcrConstants.JCR_UUID + " = '" + uuid + "']//element(*, ametys:form-entry)";
            
            String entryFilterQuery = _getEntryFilterQuery(onlyActiveEntries, additionalEntryFilterExpr);
            if (onlyActiveEntries || StringUtils.isNotBlank(entryFilterQuery))
            {
                xpathQuery += "[" + entryFilterQuery + "]";
            }
            
            String sortsAsString = "";
            for (Sort sort : sorts)
            {
                if (StringUtils.isNotBlank(sortsAsString))
                {
                    sortsAsString += ", ";
                }
                
                sortsAsString += "@ametys:" + sort.attributeName() + " " + sort.direction();
            }
            
            if (StringUtils.isNotBlank(sortsAsString))
            {
                xpathQuery += " order by " + sortsAsString;
            }
            
            AmetysObjectIterable<FormEntry> formEntries = _resolver.query(xpathQuery);
            return formEntries.stream().collect(Collectors.toList());
        }
        catch (RepositoryException e)
        {
            getLogger().error("An error occurred getting entries of form '" + form.getId() + "'");
        }
        
        return List.of();
    }
    
    private String _getEntryFilterQuery(boolean onlyActiveEntries, Expression additionalEntryFilterExpr)
    {
        List<Expression> expressions = new ArrayList<>();
        if (onlyActiveEntries)
        {
            expressions.add(new BooleanExpression(FormEntry.ATTRIBUTE_ACTIVE, true));
        }
        
        if (additionalEntryFilterExpr != null)
        {
            expressions.add(additionalEntryFilterExpr);
        }
        
        return new AndExpression(expressions).build();
    }
    
    /**
     * Get all users who answer to the form as JSON
     * @param formId the form id
     * @return all users as JSON
     */
    @Callable (rights = HANDLE_FORMS_ENTRIES_RIGHT_ID, paramIndex = 0, rightContext = FormsDirectoryRightAssignmentContext.ID)
    public List<Map<String, String>> getFormEntriesUsers(String formId)
    {
        Form form = _resolver.resolveById(formId);
        return getFormEntries(form, false, new ArrayList<>()).stream()
                .map(FormEntry::getUser)
                .distinct()
                .map(_userManager::getUser)
                .filter(Objects::nonNull)
                .sorted(Comparator.comparing(User::getFullName, String.CASE_INSENSITIVE_ORDER))
                .map(u -> Map.of("text", u.getFullName(), "id", UserIdentity.userIdentityToString(u.getIdentity())))
                .toList();
    }
    
    /**
     * Record representing a sort with form attribute name and direction
     * @param attributeName the attribute name
     * @param direction the direction
     */
    public record Sort(String attributeName, String direction) { /* */ }
}
