/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.question.sources;

import java.util.List;
import java.util.Map;

import org.ametys.plugins.forms.enumerators.LazyEnumerator;
import org.ametys.plugins.forms.repository.FormEntry;
import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.ViewItem;

/**
 * Interface for Choice source's types
 */
public interface ChoiceSourceType extends LazyEnumerator<ChoiceOption>
{
    /**
     * Get the type id
     * @return the id
     */
    public String getId();
    
    /**
     * Get the type label
     * @return the label
     */
    public I18nizableText getLabel();
    
    /**
     * Get the type model items
     * @return a map of the model items
     */
    public Map<String, ModelItem> getModelItems();
    
    /**
     * Get the view items for the main tab
     * @return a list of view items
     */
    public List<ViewItem> getViewItems();

    /**
     * Get field to disable if form is published
     * @return the list of field name
     */
    public List<String> getFieldToDisableIfFormPublished();
    
    /**
     * <code>true</code> if the enumerator has remote data and need lazy loading
     * @return <code>true</code> if the enumerator has remote data and need lazy loading
     */
    public boolean remoteData();

    /**
     * Get the storage type of the source type
     * @param question the question
     * @return the storage type
     */
    public String getStorageType(FormQuestion question);

    /**
     * Get the js renderer for the choice list
     * @return the js renderer
     */
    public String getJSRenderer();

    /**
     * Get the js converter for the choice list
     * @return the js converter
     */
    public String getJSConverter();

    /**
     * Remove the empty values or values equal to '__interval_other' from received value
     * @param value the receive value
     * @return the value without empty sub-values or without '__interval_other' if the other option is chosen. If the value is empty or equal to the other value, it returns null.
     */
    public Object removeEmptyOrOtherValue(Object value);
    
    /**
     * Convert the entry value into a JSON object to use client side
     * @param value the value to convert
     * @param question the form question
     * @param entry the form entry
     * @param modelItem the model item
     * @return The value as JSON
     * @throws Exception if an error occurred
     */
    public Object valueToJSONForClient(Object value, FormQuestion question, FormEntry entry, ModelItem modelItem) throws Exception;

    /**
     * Convert value as string
     * @param value the value
     * @return the value as string
     */
    public String value2String(Object value);

    /**
     * Get generic data used to replace answer of confidential question
     * @param question the question
     * @return the value to use as replacement
     */
    public Object getAnonimizedValue(FormQuestion question);
}
