/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.question.types.impl;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.data.holder.DataHolderRelativeDisableCondition;
import org.ametys.cms.data.holder.DataHolderRelativeDisableConditions;
import org.ametys.plugins.forms.question.sources.AbstractSourceType;
import org.ametys.plugins.forms.question.sources.ChoiceOption;
import org.ametys.plugins.forms.question.sources.ChoiceSourceType;
import org.ametys.plugins.forms.question.sources.ChoiceSourceTypeExtensionPoint;
import org.ametys.plugins.forms.question.sources.ManualSourceType;
import org.ametys.plugins.forms.question.sources.ManualWithCostsSourceType;
import org.ametys.plugins.forms.question.types.AbstractFormQuestionType;
import org.ametys.plugins.forms.question.types.MultipleAwareQuestionType;
import org.ametys.plugins.forms.repository.FormEntry;
import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.Model;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.SimpleViewItemGroup;
import org.ametys.runtime.model.StaticEnumerator;
import org.ametys.runtime.model.ViewElement;
import org.ametys.runtime.model.disableconditions.DisableCondition;
import org.ametys.runtime.model.disableconditions.DisableCondition.OPERATOR;
import org.ametys.runtime.model.disableconditions.DisableConditions;
import org.ametys.runtime.model.type.ModelItemTypeConstants;
import org.ametys.runtime.parameter.Validator;

import com.google.common.collect.Multimap;

/**
 * Class for creating choices list questions
 */
public class ChoicesListQuestionType extends AbstractFormQuestionType implements MultipleAwareQuestionType
{
    /** Constant for source type attribute */
    public static final String ATTRIBUTE_SOURCE_TYPE = "source-type";
    
    /** Constant for other option attribute */
    public static final String ATTRIBUTE_OTHER = "other-option";
    
    /** Constant for the format attribute */
    public static final String ATTRIBUTE_FORMAT = "format";
    
    /** Constant for placeholder attribute. */
    public static final String ATTRIBUTE_PLACEHOLDER = "placeholder";
    
    /** Id of the default type for formatStaticEnumerator */
    public static final String DEFAULT_TYPE_ID = "org.ametys.plugins.forms.question.sources.Manual";
    
    /** Name of checkbox formatStaticEnumerator entry */
    public static final String CHECKBOX_FORMAT_VALUE = "checkbox";
    
    /** Name of combobox formatStaticEnumerator entry */
    public static final String COMBOBOXBOX_FORMAT_VALUE = "combobox";
    
    /** Label of the other option value*/
    public static final String OTHER_OPTION_VALUE = "__internal_other";

    /** Label of the other option value*/
    public static final String OTHER_PREFIX_DATA_NAME = "ametys-other-";

    /** Constant for default title */
    public static final String DEFAULT_TITLE = "PLUGIN_FORMS_QUESTION_DEFAULT_TITLE_CHOICE_LIST";
    
    /** The choice source type extension point */
    protected ChoiceSourceTypeExtensionPoint _choiceSourceTypeExtensionPoint;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _choiceSourceTypeExtensionPoint = (ChoiceSourceTypeExtensionPoint) manager.lookup(ChoiceSourceTypeExtensionPoint.ROLE);
    }
    
    @Override
    protected List<ModelItem> _getModelItems()
    {
        List<ModelItem> modelItems = super._getModelItems();
        
        ElementDefinition<String> sourceType = _formElementDefinitionHelper.getElementDefinition(ATTRIBUTE_SOURCE_TYPE, ModelItemTypeConstants.STRING_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_CHOICES_SOURCE_TYPE", "PLUGINS_FORMS_QUESTIONS_DIALOG_CHOICES_SOURCE_TYPE_DESC", null);
        StaticEnumerator<String> typesStaticEnumerator = new StaticEnumerator<>();
        for (ChoiceSourceType type : _choiceSourceTypeExtensionPoint.getAllSourceType())
        {
            typesStaticEnumerator.add(type.getLabel(), type.getId());
            
            DisableConditions disableConditions = new DataHolderRelativeDisableConditions();
            DisableCondition condition = new DataHolderRelativeDisableCondition(ATTRIBUTE_SOURCE_TYPE, OPERATOR.NEQ, type.getId(), _disableConditionsHelper);
            disableConditions.getConditions().add(condition);
            
            Map<String, ModelItem> typeModelItems = type.getModelItems();
            for (ModelItem item : typeModelItems.values())
            {
                item.setDisableConditions(disableConditions);
            }
            
            modelItems.addAll(typeModelItems.values());
        }
        sourceType.setEnumerator(typesStaticEnumerator);
        sourceType.setDefaultValue(DEFAULT_TYPE_ID);
        modelItems.add(sourceType);
        
        modelItems.add(getMultipleModelItem());
        
        ElementDefinition<Boolean> other = _formElementDefinitionHelper.getElementDefinition(ATTRIBUTE_OTHER, ModelItemTypeConstants.BOOLEAN_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_CHOICE_OTHER", "PLUGINS_FORMS_QUESTIONS_DIALOG_CHOICE_OTHER_DESC", null);
        modelItems.add(other);
        
        ElementDefinition<String> format = _formElementDefinitionHelper.getElementDefinition(ATTRIBUTE_FORMAT, ModelItemTypeConstants.STRING_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_CHOICE_FORMAT", "PLUGINS_FORMS_QUESTIONS_DIALOG_CHOICE_FORMAT_DESC", null);
        StaticEnumerator<String> formatStaticEnumerator = new StaticEnumerator<>();
        formatStaticEnumerator.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_CHOICE_FORMAT_CHECKBOX"), CHECKBOX_FORMAT_VALUE);
        formatStaticEnumerator.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_CHOICE_FORMAT_COMBOBOX"), COMBOBOXBOX_FORMAT_VALUE);
        format.setEnumerator(formatStaticEnumerator);
        format.setDefaultValue(COMBOBOXBOX_FORMAT_VALUE);
        modelItems.add(format);
        
        ElementDefinition<String> placeholder = _formElementDefinitionHelper.getElementDefinition(ATTRIBUTE_PLACEHOLDER, ModelItemTypeConstants.STRING_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_QUESTION_PLACEHOLDER", "PLUGINS_FORMS_QUESTIONS_DIALOG_QUESTION_PLACEHOLDER_DESC", null);
        DisableConditions placeholderConditions = new DataHolderRelativeDisableConditions();
        placeholderConditions.getConditions().add(new DataHolderRelativeDisableCondition(ATTRIBUTE_FORMAT, OPERATOR.NEQ, COMBOBOXBOX_FORMAT_VALUE, _disableConditionsHelper));
        placeholder.setDisableConditions(placeholderConditions);
        modelItems.add(placeholder);
        
        return modelItems;
    }

    @Override
    protected SimpleViewItemGroup _getMainTab()
    {
        SimpleViewItemGroup mainFieldset = super._getMainTab();
        
        ViewElement format = new ViewElement();
        format.setDefinition((ElementDefinition< ? >) getModel().getModelItem(ATTRIBUTE_FORMAT));
        mainFieldset.addViewItem(format);
        
        ViewElement placeholder = new ViewElement();
        placeholder.setDefinition((ElementDefinition< ? >) getModel().getModelItem(ATTRIBUTE_PLACEHOLDER));
        mainFieldset.addViewItem(placeholder);
        
        ViewElement sourceCombobox = new ViewElement();
        sourceCombobox.setDefinition((ElementDefinition< ? >) getModel().getModelItem(ATTRIBUTE_SOURCE_TYPE));
        mainFieldset.addViewItem(sourceCombobox);
       
        for (ChoiceSourceType type : _choiceSourceTypeExtensionPoint.getAllSourceType())
        {
            mainFieldset.addViewItems(type.getViewItems());
        }
        
        return mainFieldset;
    }
    
    @Override
    protected SimpleViewItemGroup _getAdvancedTab()
    {
        SimpleViewItemGroup advancedFieldset = super._getAdvancedTab();
        
        advancedFieldset.addViewItem(getMultipleViewElement(getModel()));
        
        ViewElement other = new ViewElement();
        other.setDefinition((ElementDefinition< ? >) getModel().getModelItem(ATTRIBUTE_OTHER));
        advancedFieldset.addViewItem(other);
        
        return advancedFieldset;
    }
    
    /**
     * Get options of the choices list
     * @param question The current question
     * @return a map of the options: key is optionValue, value is optionLabel
     */
    public Map<String, I18nizableText> getOptions(FormQuestion question)
    {
        Map<String, I18nizableText> options = new LinkedHashMap<>();
        ChoiceSourceType type = _choiceSourceTypeExtensionPoint.getExtension(question.getValue(ATTRIBUTE_SOURCE_TYPE));
        if (!type.remoteData())
        {
            try
            {
                Map<String, Object> enumParam = new HashMap<>();
                enumParam.put(AbstractSourceType.QUESTION_PARAM_KEY, question);
                Map<ChoiceOption, I18nizableText> optionsToMap = type.getTypedEntries(enumParam);
                for (ChoiceOption option : optionsToMap.keySet())
                {
                    options.put((String) option.getValue(), optionsToMap.get(option));
                }
            }
            catch (Exception e)
            {
                getLogger().error("An error occured while getting options for question " + question.getId());
            }
        }
        return options;
    }
    
    @Override
    public void saxAdditionalInfos(ContentHandler contentHandler, FormQuestion question) throws SAXException
    {
        super.saxAdditionalInfos(contentHandler, question);
        ChoiceSourceType type = _choiceSourceTypeExtensionPoint.getExtension(question.getValue(ATTRIBUTE_SOURCE_TYPE));
        try
        {
            AttributesImpl attrsOption = new AttributesImpl();
            attrsOption.addCDATAAttribute("remoteData", String.valueOf(type.remoteData()));
            XMLUtils.startElement(contentHandler, "options", attrsOption);
            if (!type.remoteData())
            {
                Map<String, Object> enumParam = new HashMap<>();
                enumParam.put(AbstractSourceType.QUESTION_PARAM_KEY, question);
                Map<ChoiceOption, I18nizableText> options = type.getTypedEntries(enumParam);
                
                for (ChoiceOption option : options.keySet())
                {
                    AttributesImpl attrsImpl = new AttributesImpl();
                    Map<String, Object> attrs = option.getAttributes();
                    for (String key : attrs.keySet())
                    {
                        //ex: "cost": cost
                        attrsImpl.addCDATAAttribute(key, attrs.get(key).toString());
                    }
                    attrsImpl.addCDATAAttribute("format", question.getValue(ATTRIBUTE_FORMAT));
                    attrsImpl.addCDATAAttribute("multiple", String.valueOf(isMultiple(question)));
                    
                    XMLUtils.startElement(contentHandler, "option", attrsImpl);
                    options.get(option).toSAX(contentHandler, "label");
                    
                    XMLUtils.endElement(contentHandler, "option");
                }
            }
                
            if (hasOtherOption(question))
            {
                AttributesImpl otherAttrs = new AttributesImpl();
                otherAttrs.addCDATAAttribute("format", question.getValue(ATTRIBUTE_FORMAT));
                otherAttrs.addCDATAAttribute("multiple", String.valueOf(isMultiple(question)));
                otherAttrs.addCDATAAttribute("other", "true");
                otherAttrs.addCDATAAttribute("value", OTHER_OPTION_VALUE);
                XMLUtils.createElement(contentHandler, "option", otherAttrs);
            }
                
            XMLUtils.endElement(contentHandler, "options");
        }
        catch (Exception e)
        {
            getLogger().error("An error occurred while saxing addionnal infos of the select.", e);
        }
    }

    public String getStorageType(FormQuestion question)
    {
        ChoiceSourceType sourceType = getSourceType(question);
        if (sourceType != null)
        {
            return sourceType.getStorageType(question);
        }
        
        // Return string by default but it not supposed to have no source type
        return ModelItemTypeConstants.STRING_TYPE_ID;
    }
    
    @Override
    protected ModelItem _getEntryModelItem(FormQuestion question)
    {
        ModelItem item = super._getEntryModelItem(question);
        ((ElementDefinition) item).setMultiple(isMultiple(question));
        return item;
    }
    
    @Override
    public void doAdditionalOperations(FormQuestion question, Map<String, Object> values)
    {
        super.doAdditionalOperations(question, values);
        
        String sourceType = (String) values.get(ATTRIBUTE_SOURCE_TYPE);
        if ("org.ametys.plugins.forms.question.sources.Manual".equals(sourceType) && values.containsKey(ManualSourceType.ATTRIBUTE_GRID))
        {
            question.setValue(ManualWithCostsSourceType.ATTRIBUTE_GRID_COSTS, values.get(ManualSourceType.ATTRIBUTE_GRID));
        }
        if ("org.ametys.plugins.forms.question.sources.ManualWithCosts".equals(sourceType) && values.containsKey(ManualWithCostsSourceType.ATTRIBUTE_GRID_COSTS))
        {
            question.setValue(ManualSourceType.ATTRIBUTE_GRID, values.get(ManualWithCostsSourceType.ATTRIBUTE_GRID_COSTS));
        }
    }

    public I18nizableText getDefaultTitle()
    {
        return new I18nizableText("plugin.forms", DEFAULT_TITLE);
    }
    
    @Override
    public boolean isQuestionConfigured(FormQuestion question)
    {
        try
        {
            String format = question.getValue(ATTRIBUTE_FORMAT);
            ChoiceSourceType sourceType = getSourceType(question);
            if (sourceType.remoteData() && CHECKBOX_FORMAT_VALUE.equals(format))
            {
                return false;
            }

            Map<String, Object> enumParam = new HashMap<>();
            enumParam.put(AbstractSourceType.QUESTION_PARAM_KEY, question);
            
            if (sourceType.remoteData())
            {
                return !sourceType.searchEntries(enumParam, 1, null).isEmpty();
            }
            else
            {
                return !sourceType.getTypedEntries(enumParam).isEmpty();
            }
        }
        catch (Exception e)
        {
            getLogger().error("An error occurred getting values of source type for question with id '{}'", question.getId(), e);
            return false;
        }
    }
    
    /**
     * Get type of source for this question
     * @param question the current question
     * @return the ChoiceSourceType
     */
    public ChoiceSourceType getSourceType(FormQuestion question)
    {
        return _choiceSourceTypeExtensionPoint.getExtension(question.getValue(ATTRIBUTE_SOURCE_TYPE));
    }
    
    @Override
    public void saxEntryValue(ContentHandler contentHandler, FormQuestion question, FormEntry entry) throws SAXException
    {
        super.saxEntryValue(contentHandler, question, entry);
        
        XMLUtils.startElement(contentHandler, "additional-infos");
        
        String nameForForm = question.getNameForForm();
        AttributesImpl attr = new AttributesImpl();
        boolean hasOtherValue = hasOtherOption(question);
        attr.addCDATAAttribute("other-option", String.valueOf(hasOtherValue));
        
        XMLUtils.startElement(contentHandler, "values", attr);
        if (entry.hasValue(nameForForm))
        {
            @SuppressWarnings("cast")
            List<Object> values = entry.isMultiple(nameForForm)
                    ? Arrays.asList(entry.getValue(nameForForm))
                    : List.of((Object) entry.getValue(nameForForm)); // Need to cast in object because List.of want object and entry.getValue return typed value as string for exemple
            
            if (!values.isEmpty())
            {
                ChoiceSourceType type = _choiceSourceTypeExtensionPoint.getExtension(question.getValue(ATTRIBUTE_SOURCE_TYPE));
                try
                {
                    Map<String, Object> enumParam = new HashMap<>();
                    enumParam.put(AbstractSourceType.QUESTION_PARAM_KEY, question);
                    
                    for (Object value: values)
                    {
                        String valAsString = type.value2String(value);
                        AttributesImpl attrVal = new AttributesImpl();
                        attrVal.addCDATAAttribute("val", valAsString);
                        
                        if (type instanceof ManualWithCostsSourceType costType)
                        {
                            attrVal.addCDATAAttribute("cost", String.valueOf(costType.getEntryCost(valAsString, enumParam)));
                        }
                        
                        XMLUtils.startElement(contentHandler, "value", attrVal);
                        ChoiceOption choiceOption = new ChoiceOption(value);
                        I18nizableText label = type.getEntry(choiceOption, enumParam);
                        if (label != null)
                        {
                            label.toSAX(contentHandler, "label");
                        }
                        else
                        {
                            XMLUtils.createElement(contentHandler, "label", type.value2String(value));
                        }
                        XMLUtils.endElement(contentHandler, "value");
                    }
                }
                catch (Exception e)
                {
                    getLogger().error("An error occurred while saxing addionnal infos of the select.", e);
                }
            }
        }
        
        if (hasOtherValue && entry.hasValue(OTHER_PREFIX_DATA_NAME + nameForForm))
        {
            String value = entry.getValue(OTHER_PREFIX_DATA_NAME + nameForForm);
            AttributesImpl attrVal = new AttributesImpl();
            attrVal.addCDATAAttribute("val", value);
            attrVal.addCDATAAttribute("isOther", "true");
            
            XMLUtils.startElement(contentHandler, "value", attrVal);
            XMLUtils.createElement(contentHandler, "label", value);
            XMLUtils.endElement(contentHandler, "val");
        }

        XMLUtils.endElement(contentHandler, "values");
        XMLUtils.endElement(contentHandler, "additional-infos");
    }
    
    @Override
    public List<String> getFieldToDisableIfFormPublished(FormQuestion question)
    {
        List<String> fieldNames =  super.getFieldToDisableIfFormPublished(question);
        fieldNames.add(ATTRIBUTE_SOURCE_TYPE);
        fieldNames.add(ATTRIBUTE_MULTIPLE);
        fieldNames.add(ATTRIBUTE_OTHER);
        
        ChoiceSourceType sourceType = getSourceType(question);
        if (sourceType != null)
        {
            for (String fieldName : sourceType.getFieldToDisableIfFormPublished())
            {
                fieldNames.add(fieldName);
            }
        }
        
        return fieldNames;
    }
    
    @Override
    public void validateQuestionValues(Map<String, Object> values, Map<String, I18nizableText> errors)
    {
        super.validateQuestionValues(values, errors);
        
        String format = (String) values.get(ATTRIBUTE_FORMAT);
        String sourceTypeId = (String) values.get(ATTRIBUTE_SOURCE_TYPE);
        
        ChoiceSourceType sourceType = _choiceSourceTypeExtensionPoint.getExtension(sourceTypeId);
        if (sourceType.remoteData() && CHECKBOX_FORMAT_VALUE.equals(format))
        {
            errors.put(ATTRIBUTE_FORMAT, new I18nizableText("plugin.forms", "PLUGINS_FORMS_CHOICE_LIST_INCOMPATIBLE_ERROR"));
        }
    }

    @Override
    public Validator getMandatoryValidator(FormQuestion question)
    {
        if (hasOtherOption(question))
        {
            // Return null because for choice list, the value can be null and not the other field.
            // Its the validateEntryValues to check this
            return null;
        }
        
        return super.getMandatoryValidator(question);
    }
    
    @Override
    public void validateEntryValues(FormQuestion question, Map<String, Object> values, Multimap<String, I18nizableText> errors, Optional<Long> currentStepId, Map<String, Object> additionalParameters)
    {
        super.validateEntryValues(question, values, errors, currentStepId, additionalParameters);
        
        if (question.isMandatory() && hasOtherOption(question))
        {
            String nameForForm = question.getNameForForm();
            Object listValue = values.get(nameForForm);
            String otherValue = (String) values.get(OTHER_PREFIX_DATA_NAME + nameForForm);
            
            // Here the value can be only null if there are no value because AbstractSourceType#removeEmptyOrOtherValue is already executed
            if (listValue == null && StringUtils.isBlank(otherValue))
            {
                errors.put(nameForForm, new I18nizableText("plugin.core-ui", "PLUGINS_CORE_UI_DEFAULT_VALIDATOR_MANDATORY"));
            }
        }
    }
    
    /**
     * Get other field model
     * @param question the question
     * @return the other field model
     */
    public ModelItem getOtherFieldModel(FormQuestion question)
    {
        if (hasOtherOption(question))
        {
            return _formElementDefinitionHelper.getElementDefinition(OTHER_PREFIX_DATA_NAME + question.getNameForForm(), ModelItemTypeConstants.STRING_TYPE_ID, null, null, null);
        }
        return null;
    }
    
    /**
     * <code>true</code> if the choice list can have an other value
     * @param question the question
     * @return <code>true</code> if the choice list can have an other value
     */
    public boolean hasOtherOption(FormQuestion question)
    {
        return question.getValue(ATTRIBUTE_OTHER, false, false);
    }

    @Override
    public String getJSRenderer(FormQuestion question)
    {
        ChoiceSourceType sourceType = getSourceType(question);
        return sourceType.getJSRenderer();
    }

    @Override
    public String getJSConverter(FormQuestion question)
    {
        ChoiceSourceType sourceType = getSourceType(question);
        return sourceType.getJSConverter();
    }
    
    @Override
    public Object valueToJSONForClient(Object value, FormQuestion question, FormEntry entry, ModelItem modelItem) throws Exception
    {
        Object valueToJSONForClient = super.valueToJSONForClient(value, question, entry, modelItem);
        ChoiceSourceType sourceType = getSourceType(question);
        return sourceType.valueToJSONForClient(valueToJSONForClient, question, entry, modelItem);
    }

    public ModelItem getMultipleModelItem()
    {
        return _formElementDefinitionHelper.getElementDefinition(ATTRIBUTE_MULTIPLE, ModelItemTypeConstants.BOOLEAN_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_CHOICE_MULTIPLE", "PLUGINS_FORMS_QUESTIONS_DIALOG_CHOICE_MULTIPLE_DESC", null);
    }
    
    public ViewElement getMultipleViewElement(Model model)
    {
        ViewElement mandatory = new ViewElement();
        mandatory.setDefinition((ElementDefinition< ? >) model.getModelItem(ATTRIBUTE_MULTIPLE));
        return mandatory;
    }

    public boolean isMultiple(FormQuestion question)
    {
        return question.getValue(ATTRIBUTE_MULTIPLE, false, false);
    }
    
    @Override
    public Map<String, Object> getAnonymizedData(FormQuestion question)
    {
        if (hasOtherOption(question))
        {
            Map<String, Object> result = new HashMap<>(2);
            result.put(question.getNameForForm(), null);
            result.put(OTHER_PREFIX_DATA_NAME + question.getNameForForm(), isMandatory(question) ? "anonymized" : null);
            return result;
        }
        else
        {
            ChoiceSourceType sourceType = getSourceType(question);
            return Collections.singletonMap(question.getNameForForm(), isMandatory(question) ? sourceType.getAnonimizedValue(question) : null);
        }
    }
}
