/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.repository.type;

import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Map.Entry;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.core.model.type.AbstractElementType;
import org.ametys.plugins.repository.data.repositorydata.ModifiableRepositoryData;
import org.ametys.plugins.repository.data.repositorydata.RepositoryData;
import org.ametys.plugins.repository.data.type.RepositoryElementType;
import org.ametys.runtime.model.exception.BadItemTypeException;
import org.ametys.runtime.model.type.DataContext;

/**
 * Class for matrix element types
 */
public class MatrixElementType extends AbstractElementType<Matrix> implements RepositoryElementType<Matrix>
{
    /** Constant for type matrix */
    public static final String MATRIX_DATA_TYPE = "matrix";
    
    public String getRepositoryDataType()
    {
        return RepositoryData.STRING_REPOSITORY_DATA_TYPE;
    }

    @Override
    public Matrix convertValue(Object value)
    {
        if (value == null)
        {
            return null;
        }
        else if (value instanceof String)
        {
            return _string2Matrix((String) value);
        }
        else if (value instanceof Map)
        {
            @SuppressWarnings("unchecked")
            Map<String, List<String>> castMap = (Map<String, List<String>>) value;
            return new Matrix(castMap);
        }
        else
        {
            throw new BadItemTypeException("Try to convert the non " + getId() + " JSON object '" + value + "' into a " + getId());
        }
    }
    
    @Override
    public String toString(Matrix value)
    {
        return _jsonUtils.convertObjectToJson(value);
    }

    @Override
    public boolean isSimple()
    {
        return false;
    }
    
    @Override
    public boolean isCompatible(Object value)
    {
        return super.isCompatible(value)
            || value instanceof String || value instanceof String[]
            || value instanceof Matrix || value instanceof Matrix[];
    }
    
    /**
     * Map json to matrix
     * @param value the json value
     * @return the matrix
     */
    @SuppressWarnings("unchecked")
    protected Matrix _string2Matrix(String value)
    {
        Map<String, List<String>> castMap = _jsonUtils.convertJsonToMap(value)
            .entrySet()
            .stream()
            .collect(Collectors.toMap(e -> e.getKey(), e -> (List<String>) e.getValue(), (e1, e2) -> e1, LinkedHashMap::new));
        return new Matrix(castMap);
    }
    
    @Override
    protected void _singleTypedNotEnumeratedValueToSAX(ContentHandler contentHandler, String tagName, Matrix value, DataContext context, AttributesImpl attributes)
            throws SAXException
    {
        AttributesImpl localAttributes = new AttributesImpl(attributes);
        
        XMLUtils.startElement(contentHandler, tagName, localAttributes);
        
        for (Entry<String, List<String>> lineValue : value.entrySet())
        {
            AttributesImpl optAttr = new AttributesImpl();
            optAttr.addCDATAAttribute("value", lineValue.getKey());
            XMLUtils.startElement(contentHandler, "option", optAttr);
            List<String> colValues = lineValue.getValue();
            for (String colValue : colValues)
            {
                XMLUtils.createElement(contentHandler, "value", colValue);
            }
            XMLUtils.endElement(contentHandler, "option");
        }
        
        XMLUtils.endElement(contentHandler, tagName);
    }

    public Object read(RepositoryData parentData, String name) throws BadItemTypeException
    {
        if (!parentData.hasValue(name))
        {
            return null;
        }
        
        if (!isCompatible(parentData, name))
        {
            throw new BadItemTypeException("Try to get " + getId() + " value from the non " + getId() + " data '" + name + "' on '" + parentData + "'");
        }
        
        if (parentData.isMultiple(name))
        {
            String[] values = parentData.getStrings(name);
            return Arrays.stream(values)
                        .map(this::_string2Matrix)
                        .toArray(Matrix[]::new);
        }
        else
        {
            String value = parentData.getString(name);
            return _string2Matrix(value);
        }
    }
    
    public boolean hasNonEmptyValue(RepositoryData parentData, String name) throws BadItemTypeException
    {
        if (!parentData.hasValue(name))
        {
            return false;
        }
        
        if (!isCompatible(parentData, name))
        {
            throw new BadItemTypeException("Try to check matrix value from the non " + getId() + " data '" + name + "' on '" + parentData + "'");
        }
        
        if (parentData.isMultiple(name))
        {
            return parentData.getStrings(name).length > 0;
        }
        else
        {
            return StringUtils.isNotEmpty(parentData.getString(name));
        }
    }

    public void write(ModifiableRepositoryData parentData, String name, Object value) throws BadItemTypeException
    {
        if (value == null)
        {
            if (parentData.hasValue(name) && parentData.isMultiple(name))
            {
                parentData.setValues(name, new String[0]);
            }
            else
            {
                parentData.setValue(name, StringUtils.EMPTY);
            }
        }
        else if (value instanceof String)
        {
            parentData.setValue(name, (String) value);
        }
        else if (value instanceof String[])
        {
            Arrays.stream((String[]) value)
                  .forEach(v -> Optional.ofNullable(v)
                                        .orElseThrow(() -> new IllegalArgumentException("Try to set a null value into the multiple " + getId() + " data '" + name + "' on '" + parentData + "'")));
            
            parentData.setValues(name, (String[]) value);
        }
        else if (value instanceof Matrix)
        {
            Matrix matrix = (Matrix) value;
            parentData.setValue(name, _jsonUtils.convertObjectToJson(matrix));
        }
        else if (value instanceof Matrix[])
        {
            String[] values = Stream.of((Matrix[]) value)
                .map(m -> _jsonUtils.convertObjectToJson(m))
                .toArray(String[]::new);
            
            parentData.setValues(name, values);
        }
        else if (value instanceof List)
        {
            @SuppressWarnings("unchecked")
            List<Matrix> matrixList = (List<Matrix>) value;
            String[] matrixListAsJson = matrixList.stream()
                .map(m -> _jsonUtils.convertObjectToJson(m))
                .toArray(String[]::new);
            
            parentData.setValues(name, matrixListAsJson);
        }
        else
        {
            throw new BadItemTypeException("Try to set the non " + getId() + " value '" + value + "' to the " + getId() + " data '" + name + "' on '" + parentData + "'");
        }
        
    }
}
