/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.workflow;

import java.io.IOException;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.cms.workflow.history.DescriptionStep;
import org.ametys.cms.workflow.history.ElementWithWorkflow;
import org.ametys.cms.workflow.history.HistoryStep;
import org.ametys.cms.workflow.history.WorkflowAction;
import org.ametys.cms.workflow.history.WorkflowHistory;
import org.ametys.cms.workflow.history.WorkflowHistoryHelper;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.DateUtils;
import org.ametys.core.util.URIUtils;
import org.ametys.plugins.core.user.UserHelper;
import org.ametys.plugins.forms.FormsException;
import org.ametys.plugins.forms.data.Answer;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.workflow.support.WorkflowHelper;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.WebConstants;
import org.ametys.web.WebHelper;
import org.ametys.web.repository.page.ZoneItem;

import com.opensymphony.workflow.Workflow;

/**
 * This class generates all the forms process information for current user
 */
public abstract class AbstractFormDashboardGenerator extends ServiceableGenerator
{
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    /** The ametys object resolver */
    protected WorkflowHistoryHelper _workflowHistoryHelper;
    
    /** The workflow helper component */
    protected WorkflowHelper _workflowHelper;
    
    /** The user helper */
    protected UserHelper _userHelper;
    
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _currentUserProvider = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);
        _workflowHistoryHelper = (WorkflowHistoryHelper) smanager.lookup(WorkflowHistoryHelper.ROLE);
        _workflowHelper = (WorkflowHelper) smanager.lookup(WorkflowHelper.ROLE);
        _userHelper = (UserHelper) smanager.lookup(UserHelper.ROLE);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
    }
    
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        contentHandler.startDocument();
        
        UserIdentity user = _currentUserProvider.getUser();
        if (user == null)
        {
            getLogger().warn("No user is connected. Return no anwsers.");
            XMLUtils.createElement(contentHandler, "no-user-connected");
        }
        else
        {
            _saxFormDashboardData(user);
        }
        
        contentHandler.endDocument();
    }
    
    /**
     * Sax the form dashboard data
     * @param user the current user
     * @throws SAXException if a sax exception occurred
     */
    protected void _saxFormDashboardData(UserIdentity user) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "answers");
        _saxAnswers(user);
        XMLUtils.endElement(contentHandler, "answers");
    }
    
    /**
     * Sax the answers
     * @param user the user
     */
    protected void _saxAnswers(UserIdentity user)
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        String siteName = WebHelper.getSiteName(request);
        String lang = (String) request.getAttribute("sitemapLanguage");
        
        ZoneItem zoneItem = (ZoneItem) request.getAttribute(WebConstants.REQUEST_ATTR_ZONEITEM);
        
        if (zoneItem == null)
        {
            String zoneItemId = request.getParameter("zoneItemId");
            zoneItem = _resolver.resolveById(zoneItemId);
        }
        
        boolean showAnswerWithWorkflow = zoneItem.getServiceParameters().getValue("no-workflow", false, false);
        
        try
        {
            List<Answer> answers = _getAnswers(request, siteName, lang, user);
            if (!showAnswerWithWorkflow)
            {
                answers = answers.stream()
                        .filter(a -> StringUtils.isNotBlank(a.getWorkflowName()))
                        .collect(Collectors.toList());
            }
            
            for (Answer answer : answers)
            {
                _saxAnswer(answer, user);
            }
        }
        catch (Exception e) 
        {
            getLogger().error("An error occurred saxing answers", e);
        }
    }
    
    /**
     * Sax the answer
     * @param answer the answer
     * @param user the user
     * @throws Exception if an error occurred
     */
    protected void _saxAnswer(Answer answer, UserIdentity user) throws Exception
    {
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("formId", answer.getFormId());
        attrs.addCDATAAttribute("formLabel", URIUtils.decode(answer.getFormLabel()));
        attrs.addCDATAAttribute("creationDate", DateUtils.dateToString(answer.getCreationDate()));
        attrs.addCDATAAttribute("id", answer.getId());
        attrs.addCDATAAttribute("number", String.valueOf(answer.getNumber()));
        attrs.addCDATAAttribute("user", _userHelper.getUserFullName(answer.getUser()));
        
        XMLUtils.startElement(contentHandler, "answer", attrs);
        _saxWorkflowInformations(answer, user);
        XMLUtils.endElement(contentHandler, "answer");
    }
    
    /**
     * Get the list of answers to sax
     * @param request the request
     * @param siteName the site name
     * @param lang the lang
     * @param user the user
     * @return the list of common answer
     * @throws FormsException if a form error occurred
     */
    protected abstract List<Answer> _getAnswers(Request request, String siteName, String lang, UserIdentity user) throws FormsException;
    
    /**
     * Get the workflow of answer
     * @param answer the answer
     * @return the workflow
     */
    protected abstract Workflow _getWorkflow(Answer answer);
    
    /**
     * Sax the workflow informations
     * @param answer the answer
     * @param user the user
     * @throws Exception if an error occurred
     */
    protected void _saxWorkflowInformations(Answer answer, UserIdentity user) throws Exception
    {
        Integer workflowId = answer.getWorkflowId();
        if (workflowId != null && workflowId != -1)
        {
            ElementWithWorkflow element = new ElementWithWorkflow(
                answer.getFormLabel(),
                DateUtils.asZonedDateTime(answer.getCreationDate()),
                answer.getUser()
            );
            
            int initialActionId = _workflowHelper.getInitialAction(answer.getWorkflowName()); 
            WorkflowHistory workflowHistory = _workflowHistoryHelper.getWorkflowHistory(element, workflowId, _getWorkflow(answer), initialActionId);
            
            for (HistoryStep step : workflowHistory.getSteps())
            {
                AttributesImpl stepAttrs = new AttributesImpl();
                Optional<String> comments = step.getComments();
                if (comments.isPresent())
                {
                    stepAttrs.addCDATAAttribute("comments", comments.get());
                }
                
                stepAttrs.addCDATAAttribute("date", DateUtils.dateToString(step.getDate()));
                stepAttrs.addCDATAAttribute("isCurrent", String.valueOf(step.isCurrent()));
                stepAttrs.addCDATAAttribute("id", String.valueOf(step.getId()));
                XMLUtils.startElement(contentHandler, "step", stepAttrs);
                _saxDescription(step);
                _saxAction(step);
                _saxCaller(step);
                XMLUtils.endElement(contentHandler, "step");
            }
        }
    }
    
    private void _saxDescription(HistoryStep step) throws SAXException
    {
        DescriptionStep description = step.getDescription();
        AttributesImpl descAttrs = new AttributesImpl();
        descAttrs.addCDATAAttribute("icon", description.getMediumIcon().get());
        descAttrs.addCDATAAttribute("status", description.getStatus());
        XMLUtils.startElement(contentHandler, "description", descAttrs);
        I18nizableText desc = description.getDescription();
        desc.toSAX(contentHandler, "label");
        XMLUtils.endElement(contentHandler, "description");
    }
    
    private void _saxAction(HistoryStep step) throws SAXException
    {
        WorkflowAction workflowAction = step.getWorkflowAction();
        AttributesImpl actionAttrs = new AttributesImpl();
        actionAttrs.addCDATAAttribute("icon", workflowAction.getMediumIcon().get());
        XMLUtils.startElement(contentHandler, "action", actionAttrs);
        I18nizableText label = workflowAction.getLabel();
        label.toSAX(contentHandler, "label");
        XMLUtils.endElement(contentHandler, "action");
    }
    
    private void _saxCaller(HistoryStep step) throws SAXException
    {
        Optional<UserIdentity> caller = step.getCaller();
        if (caller.isPresent())
        {
            _userHelper.saxUserIdentity(caller.get(), contentHandler, "caller");
        }
    }
}
