/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.repository;

import javax.jcr.Node;
import javax.jcr.RepositoryException;

import org.ametys.plugins.forms.FormAndDirectoryCommonMethods;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.MovableAmetysObject;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.RepositoryIntegrityViolationException;
import org.ametys.plugins.repository.jcr.DefaultTraversableAmetysObject;
import org.ametys.web.repository.SiteAwareAmetysObject;
import org.ametys.web.repository.site.Site;

/**
 * Class representing a form directory, backed by a JCR node.<br>
 */
public class FormDirectory extends DefaultTraversableAmetysObject<FormDirectoryFactory> implements MovableAmetysObject, SiteAwareAmetysObject
{
    /** Constant for title metadata. */
    public static final String ATTRIBUTE_TITLE = RepositoryConstants.NAMESPACE_PREFIX + ":title";
    
    /**
     * Creates a {@link FormDirectory}.
     * @param node the node backing this {@link AmetysObject}
     * @param parentPath the parentPath in the Ametys hierarchy
     * @param factory the DefaultAmetysObjectFactory which created the AmetysObject
     */
    public FormDirectory(Node node, String parentPath, FormDirectoryFactory factory)
    {
        super(node, parentPath, factory);
    }
    
    /**
     * Retrieves the title.
     * @return the title.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public String getTitle() throws AmetysRepositoryException
    {
        try
        {
            if (getNode().hasProperty(ATTRIBUTE_TITLE))
            {
                return getNode().getProperty(ATTRIBUTE_TITLE).getString();
            }
            else
            {
                return getNode().getName();
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get title property", e);
        }
    }

    /**
     * Set the title.
     * @param title the title.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void setTitle(String title) throws AmetysRepositoryException
    {
        try
        {
            getNode().setProperty(ATTRIBUTE_TITLE, title);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set title property", e);
        }
    }

    public void moveTo(AmetysObject newParent, boolean renameIfExist) throws AmetysRepositoryException, RepositoryIntegrityViolationException
    {
        FormAndDirectoryCommonMethods.moveTo(newParent, renameIfExist, this);
    }

    public boolean canMoveTo(AmetysObject newParent) throws AmetysRepositoryException
    {
        return FormAndDirectoryCommonMethods.canMoveTo(getSiteName(), newParent, this, _getFactory().getFormDirectoryDAO());
    }

    public void orderBefore(AmetysObject siblingNode) throws AmetysRepositoryException
    {
        FormAndDirectoryCommonMethods.orderBefore(siblingNode, this);
    }
    
    public String getSiteName() throws AmetysRepositoryException
    {
        return getSite().getName();
    }

    public Site getSite() throws AmetysRepositoryException
    {
        AmetysObject parent = getParent();
        while (parent != null && !(parent instanceof Site)) 
        {
            parent = parent.getParent();
        }
        if (parent == null)
        {
            throw new AmetysRepositoryException("An error occurred with form with id '" + getId() + "'. Form directories must always be linked to a site");
        }
        return (Site) parent;
    }
}
