/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * Class for creating & managing Acknowledgement of receipt's dialog box
 */
 Ext.define('Ametys.plugins.forms.entries.EntriesReceiptDialog', {
	    singleton: true,
		
		/**
	     * Open the dialog box to add acknowledgement of receipt
	     * @param {Ametys.plugins.forms.controllers.EntryReceiptController} controller the controller calling this function
	     */
		act: function(controller)
		{
			var target = Ametys.message.MessageTargetHelper.findTarget(controller.getMatchingTargets(), Ametys.message.MessageTarget.FORM_TARGET);
	        if (!target)
	        {
	            return;
	        }
	        
	        this._formId = target.getParameters().id;
	        Ametys.data.ServerComm.callMethod({
                role: "org.ametys.plugins.forms.helper.AcknowledgementOfReceiptsHelper",
                methodName: "getAcknowledgementReceiptProperties",
                parameters: [this._formId],
                callback: {
                    handler: this._actCB,
                    scope: this
                },
                waitMessage: false,
                errorMessage: true
            });      
		},
		
		/**
         * @private
         * The callback function after getting acknowledgement receipt properties
         * @param {Object} response the response
         * @param {Object} args the arguments
         */
		_actCB: function(response, args)
		{
            var initConfig = {
                initFn: Ext.bind(this._initForm, this, [response], 1),
                title: "{{i18n PLUGINS_FORMS_ACKNOWLEDGEMENT_RECEIPT_DIALOG_TITLE}}",
                subject: "{{i18n PLUGINS_FORMS_ACKNOWLEDGEMENT_RECEIPT_SUBJECT_DEFAULT}}",
                body: "{{i18n PLUGINS_FORMS_ACKNOWLEDGEMENT_RECEIPT_BODY_DEFAULT}}"
            };
            
            var validConfig = {
                validFn: Ext.bind(this._ok, this)
            };
            
            var removeConfig = {
                removeFn: Ext.bind(this._remove, this)
            };
            
            Ametys.plugins.forms.helper.FormMailDialogHelper.open(this._formId, initConfig, validConfig, removeConfig)
        },
		
		/**
		 * @private
		 * Initialize the form with values provided by the server
		 * @param {Function} cbFunction the callback function
		 * @param {Object} response the response
		 */
		_initForm: function (cbFunction, response)
		{
            var params = {};
            params.sender = response["receipt-sender"];
            params.receiver = response["receipt-receiver"];
            params.subject = response["receipt-subject"];
            params.body = response["receipt-body"];
            
            cbFunction(params);
		},
		
		/**
         * @private
         * Function to call to set receipts
         * @param {String} sender the sender
         * @param {String} receiver the receiver
         * @param {String} subject the subject
         * @param {String} body the body
         * @param {Function} cbFunction the callback function
         */
		_ok: function(sender, receiver, subject, body, cbFunction)
		{
			Ametys.data.ServerComm.callMethod({
				role: "org.ametys.plugins.forms.helper.AcknowledgementOfReceiptsHelper",
	            methodName: "setAcknowledgementReceiptProperties",
	            parameters: [this._formId, sender, receiver, subject, body],
	            callback: {
	                handler: this._setOrRemoveReceiptCb,
	                scope: this,
	                arguments: {
                        cb: cbFunction
                    }
	            },
	            waitMessage: true,
                errorMessage: true
	        });    
		},
		
		/**
		 * @private
		 * Process for removing acknowledgement of receipt 
		 * @param {Function} cbFunction the callback function
		 */
		_remove: function (cbFunction)
		{
	        Ametys.data.ServerComm.callMethod({
				role: "org.ametys.plugins.forms.helper.AcknowledgementOfReceiptsHelper",
	            methodName: "removeEntriesReceipt",
	            parameters: [this._formId],
	            callback: {
	                handler: this._setOrRemoveReceiptCb,
	                scope: this,
	                arguments: {
                        cb: cbFunction
                    }
	            },
	            waitMessage: true,
                errorMessage: true
	        });
		},
		
		/**
         * @private
         * Callback function after setting of removing receipt params
         * @param {Object} response the server's response
         * @param {Object} args the arguments
         */
        _setOrRemoveReceiptCb: function (response, args)
        {
            Ext.create('Ametys.message.Message', {
                type: Ametys.message.Message.MODIFIED,
                targets: {
                    id: Ametys.message.MessageTarget.FORM_TARGET,
                    parameters: {
                        id: this._formId
                    }
                }
            });
            
            args.cb();
        }
	}
);
