/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.actions;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;

import org.ametys.core.cocoon.JSonReader;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.forms.helper.FormStatisticsHelper;
import org.ametys.plugins.forms.helper.LimitedEntriesHelper;
import org.ametys.plugins.forms.helper.ScheduleOpeningHelper;
import org.ametys.plugins.forms.helper.ScheduleOpeningHelper.FormStatus;
import org.ametys.plugins.forms.repository.Form;
import org.ametys.plugins.forms.repository.FormEntry;
import org.ametys.plugins.repository.AmetysObjectResolver;

/**
 * Action to get dynamic information about a survey
 */
public class GetDynamicFormInformationAction extends ServiceableAction
{
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;

    /** The limited entries helper */
    protected LimitedEntriesHelper _limitedEntriesHelper;
    
    /** The form statistics helper */
    protected FormStatisticsHelper _formStatsHelper;
    
    /** The schedule opening helper */
    protected ScheduleOpeningHelper _scheduleOpeningHelper;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _currentUserProvider = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);
        _limitedEntriesHelper = (LimitedEntriesHelper) smanager.lookup(LimitedEntriesHelper.ROLE);
        _formStatsHelper = (FormStatisticsHelper) smanager.lookup(FormStatisticsHelper.ROLE);
        _scheduleOpeningHelper = (ScheduleOpeningHelper) smanager.lookup(ScheduleOpeningHelper.ROLE);
    }
    
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Map<String, Object> result = new HashMap<>();
        
        Request request = ObjectModelHelper.getRequest(objectModel);
        
        String formId = parameters.getParameter("id", request.getParameter("id"));
        Form form = _resolver.resolveById(formId);
        if (form.isMiniSurvey())
        {
            // Number of votes
            List<FormEntry> entries = form.getEntries(); 
            result.put("nbEntries", entries.size());
            
            // Status
            FormStatus status = _scheduleOpeningHelper.getStatus(form);
            result.put("status", status.name());
    
            // Did the user answer to the survey?
            String clientIp = _limitedEntriesHelper.getClientIp(request);
            UserIdentity currentUser = _currentUserProvider.getUser();
            boolean answered = _limitedEntriesHelper.hasUserAlreadyAnswer(form, currentUser, clientIp);
            result.put("answered", answered);
    
            // Results
            if (answered || FormStatus.OVER == status)
            {
                result.put("results", _formStatsHelper.getMiniSurveyStatistics(form.getId()));
            }
        }
        else
        {
            result.put("results", Map.of("error", "not-a-mini-survey"));
        }
        
        request.setAttribute(JSonReader.OBJECT_TO_READ, result);
        
        return EMPTY_MAP;
    }
}
