/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.content.workflow;

import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import javax.jcr.Node;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.environment.Request;

import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.forms.FormsException;
import org.ametys.plugins.forms.content.Form;
import org.ametys.plugins.forms.content.jcr.FormPropertiesManager;
import org.ametys.plugins.forms.content.table.FormTableManager;
import org.ametys.plugins.forms.data.Answer;
import org.ametys.plugins.forms.workflow.AbstractFormDashboardGenerator;
import org.ametys.plugins.workflow.store.JdbcWorkflowStore;
import org.ametys.plugins.workflow.support.WorkflowProvider;

import com.opensymphony.workflow.Workflow;

/**
 * This class generates all the forms process information for current user
 */
public class FormDashboardGenerator extends AbstractFormDashboardGenerator
{
    /** The form table manager */
    protected FormTableManager _formTableManager;
    
    /** The form properties manager */
    protected FormPropertiesManager _formPropertiesManager;
    
    /** The workflow provider */
    protected WorkflowProvider _workflowProvider;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _formTableManager = (FormTableManager) smanager.lookup(FormTableManager.ROLE);
        _formPropertiesManager = (FormPropertiesManager) smanager.lookup(FormPropertiesManager.ROLE);
        _workflowProvider = (WorkflowProvider) smanager.lookup(WorkflowProvider.ROLE);
    }
    
    @Override
    protected List<Answer> _getAnswers(Request request, String siteName, String lang, UserIdentity user) throws FormsException
    {
        List<Form> forms = _formPropertiesManager.getFormContentNodes(siteName, lang)
                .stream()
                .map(this::_getForms)
                .flatMap(List::stream)
                .collect(Collectors.toList());
        
        return _formTableManager.getUserAnwsers(forms, user);
    }
    
    @Override
    protected Workflow _getWorkflow(Answer answer)
    {
        return _workflowProvider.getExternalWorkflow(JdbcWorkflowStore.ROLE);
    }
    
    private List<Form> _getForms(Node node)
    {
        try
        {
            return _formPropertiesManager.getForms(node);
        }
        catch (Exception e)
        {
            getLogger().error("Can get forms from content node", e);
        }
        
        return Collections.EMPTY_LIST;
    }
}
