/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.helper;

import java.util.function.Function;

import org.apache.avalon.framework.component.Component;
import org.apache.cocoon.sitemap.PatternException;
import org.apache.commons.lang3.StringUtils;

import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * The parser for mail variables
 */
public class MailVariableParser extends AbstractLogEnabled implements Component
{
    /** Avalon ROLE. */
    public static final String ROLE = MailVariableParser.class.getName();
    
    /**
     * Check expression for variables inside '{}' and replace them with their processed value
     * @param expression the epression to check
     * @param function the function processing the variables
     * @return the expression with replaced variable
     * @throws PatternException exception when braces are in a wrong space
     */
    public String replaceText(String expression, Function<String, String> function) throws PatternException
    {
        if (StringUtils.isBlank(expression))
        {
            return null;
        }
        
        StringBuilder sb = new StringBuilder();
        boolean isOpen = false;
        int openPos = 0;
        for (int i = 0; i < expression.length(); i++)
        {
            final char c = expression.charAt(i);
            
            if (c == '\\')
            {
                i++; //next character is ignored because it's escaped
                sb.append(expression.charAt(i));
            }
            else if (c == '{')
            {
                if (isOpen)
                {
                    throw new PatternException("Invalid '{' at position " + i + " in expression \"" + expression + "\"");
                }
                else
                {
                    isOpen = true;
                    openPos = i;
                }
            }
            else if (c == '}')
            {
                if (!isOpen)
                {
                    throw new PatternException("Invalid '}' at position " + i + " in expression \"" + expression + "\"");
                }
                else
                {
                    String variable = expression.substring(openPos + 1, i);
                    String translation = function.apply(variable);
                    if (StringUtils.isNotBlank(translation))
                    {
                        sb.append(translation);
                    }
                    isOpen = false;
                }
            }
            else if (!isOpen)
            {
                sb.append(c);
            }
        }
        
        if (isOpen)
        {
            throw new PatternException("Invalid '{' at position " + openPos + " in expression \"" + expression + "\"");
        }
        
        return sb.toString();
    }
}
