/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.question.validators;

import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.time.format.FormatStyle;
import java.util.HashMap;
import java.util.Map;

import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;

/**
 * A local date interval form validator
 */
public class LocalDateIntervalFormValidator extends AbstractIntervalFormValidator<LocalDate>
{
    /**
     * The constructor
     * @param regexp the regexp
     * @param mandatory mandatory
     * @param min the min value
     * @param max the max value
     */
    public LocalDateIntervalFormValidator(String regexp, boolean mandatory, LocalDate min, LocalDate max)
    {
        super(regexp, mandatory, min, max);
    }

    @Override
    protected boolean isLessThan(LocalDate dateToValidate, LocalDate dateMin)
    {
        return dateToValidate.isBefore(dateMin);
    }

    @Override
    protected boolean isMoreThan(LocalDate dateToValidate, LocalDate dateMax)
    {
        return dateToValidate.isAfter(dateMax);
    }

    @Override
    protected I18nizableText getIntervalError(LocalDate min, LocalDate max)
    {
        Map<String, I18nizableTextParameter> i18nParams = new HashMap<>();
        i18nParams.put("startDate", new I18nizableText(min.format(DateTimeFormatter.ofLocalizedDate(FormatStyle.SHORT))));
        i18nParams.put("endDate", new I18nizableText(max.format(DateTimeFormatter.ofLocalizedDate(FormatStyle.SHORT))));
        return new I18nizableText("plugin.forms", "PLUGINS_FORMS_ENTRY_VALIDATOR_DATE_INTERVAL_ERROR", i18nParams);
    }
}
