/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.repository;

import javax.jcr.Node;

import org.ametys.cms.data.ametysobject.ModifiableModelAwareDataAwareAmetysObject;
import org.ametys.cms.data.holder.ModifiableIndexableDataHolder;
import org.ametys.cms.data.holder.impl.DefaultModifiableModelAwareDataHolder;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.data.repositorydata.ModifiableRepositoryData;
import org.ametys.plugins.repository.data.repositorydata.impl.JCRRepositoryData;
import org.ametys.plugins.repository.jcr.SimpleAmetysObject;

/**
 * Class representing a form rule on a page.
 */
public class FormPageRule extends SimpleAmetysObject<FormPageRuleFactory> implements ModifiableModelAwareDataAwareAmetysObject
{
    /** Constant for the date and time of submission attribute. */
    public static final String ATTRIBUTE_OPTION_VALUE = "option-value";
    
    /** Constant for the rule type attribute. */
    public static final String ATTRIBUTE_TYPE = "type";
    
    /** Constant for the page id attribute. */
    public static final String ATTRIBUTE_TARGET_ID = "target-id";
    
    /** Type of a page rule. */
    public enum PageRuleType
    {
        /** Jump. */
        JUMP,
        /** Skip. */
        SKIP,
        /** Finish. */
        FINISH
    }
    
    /**
     * Creates an {@link FormPageRule}.
     * @param node the node backing this {@link AmetysObject}
     * @param parentPath the parentPath in the Ametys hierarchy
     * @param factory the DefaultAmetysObjectFactory which created the AmetysObject
     */
    public FormPageRule(Node node, String parentPath, FormPageRuleFactory factory)
    {
        super(node, parentPath, factory);
    }
    
    /**
     * Get the type
     * @return the type
     */
    public PageRuleType getType ()
    {
        String value = getValue(ATTRIBUTE_TYPE);
        return PageRuleType.valueOf(value);
    }
    
    /**
     * Set the type
     * @param type the rule type
     */
    public void setType (PageRuleType type)
    {
        setValue(ATTRIBUTE_TYPE, type.name());
    }
    
    /**
     * Get the page to jump or skip. Can be null.
     * @return the page to jump or skip
     */
    public String getPageId ()
    {
        return getValue(ATTRIBUTE_TARGET_ID);
    }

    /**
     * Set the page id
     * @param id the page id
     */
    public void setPageId (String id)
    {
        setValue(ATTRIBUTE_TARGET_ID, id);
    }
    
    /**
     * Get the selected option
     * @return the selected option
     */
    public String getOption ()
    {
        return getValue(ATTRIBUTE_OPTION_VALUE);
    }
    
    /**
     * Set the option value
     * @param value the option value
     */
    public void setOption(String value)
    {
        setValue(ATTRIBUTE_OPTION_VALUE, value);
    }
    
    @Override
    public ModifiableIndexableDataHolder getDataHolder()
    {
        ModifiableRepositoryData repositoryData = new JCRRepositoryData(getNode());
        return new DefaultModifiableModelAwareDataHolder(repositoryData, _getFactory().getFormRuleModel());
    }
}
