/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.repository.type;

import java.util.HashMap;
import java.util.Map;

import org.ametys.runtime.i18n.I18nizableText;

/**
 * Object representing a rule
 */
public class Rule 
{
    private String _sourceId;
    private QuestionRuleType _action;
    private String _option;
    
    /**
     * The Rule constructor
     * @param id the source question's id
     * @param option the trigger option's value
     * @param type the action on the target when triggered
     */
    public Rule(String id, String option, QuestionRuleType type)
    {
        setSourceId(id);
        setAction(type);
        setOption(option);
    }

    /** Type of a question rule. */
    public enum QuestionRuleType
    {
        /** Hide. */
        HIDE, 
        /** Show. */
        SHOW;
        
        /** Get an i18nizable label for the RuleType
         * @return a label as an I180nizableText
         */
        public I18nizableText getLabel()
        {
            switch (this)
            {
                case HIDE:
                    return new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTION_RULES_ACTION_HIDE");
                case SHOW:
                    return new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTION_RULES_ACTION_SHOW");
                default:
                    throw new IllegalArgumentException(this + " is not a known type question rule type");
            }
        }
    }
    
    /**
     * Get the source question's id
     * @return the question's id
     */
    public String getSourceId()
    {
        return _sourceId;
    }

    /**
     * Set the source question's id
     * @param questionId the question's id
     */
    public void setSourceId(String questionId)
    {
        this._sourceId = questionId;
    }

    /**
     * Get the action on the target when triggered
     * @return the Rule's action
     */
    public QuestionRuleType getAction()
    {
        return _action;
    }

    /**
     * Set the Question Rule type 
     * @param type the type of _action to set
     */
    public void setAction(QuestionRuleType type)
    {
        this._action = type;
    }

    /**
     * Get the option value that trigger the action 
     * @return the _option
     */
    public String getOption()
    {
        return _option;
    }

    /**
     * Set the option value that trigger the action 
     * @param option the _option to set
     */
    public void setOption(String option)
    {
        this._option = option;
    }
    
    /**
     * Get the rule attributes formated for json export
     * @return a json readable map 
     */
    public Map<String, Object> toJson()
    {
        Map<String, Object> json = new HashMap<>();
        
        json.put("question", this._sourceId);
        json.put("option", this._option);
        json.put("action", this._action.name());
        
        return json;
    }
}
