/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.helper;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.stream.Collectors;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.right.RightManager;
import org.ametys.core.ui.Callable;
import org.ametys.core.util.I18nUtils;
import org.ametys.plugins.forms.dao.FormEntryDAO;
import org.ametys.plugins.forms.question.FormQuestionType;
import org.ametys.plugins.forms.question.sources.AbstractSourceType;
import org.ametys.plugins.forms.question.sources.ChoiceOption;
import org.ametys.plugins.forms.question.sources.ChoiceSourceType;
import org.ametys.plugins.forms.question.types.impl.CheckBoxQuestionType;
import org.ametys.plugins.forms.question.types.impl.ChoicesListQuestionType;
import org.ametys.plugins.forms.question.types.impl.MatrixQuestionType;
import org.ametys.plugins.forms.repository.Form;
import org.ametys.plugins.forms.repository.FormEntry;
import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.plugins.forms.repository.type.Matrix;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.BooleanExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.MetadataExpression;
import org.ametys.plugins.repository.query.expression.NotExpression;
import org.ametys.plugins.repository.query.expression.OrExpression;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * The helper to handle admin emails
 */
public class FormStatisticsHelper extends AbstractLogEnabled implements Serviceable, Component
{
    /** Avalon ROLE. */
    public static final String ROLE = FormStatisticsHelper.class.getName();
    
    /** The Ametys Object resolver */
    protected AmetysObjectResolver _resolver;

    /** The right manager */
    protected RightManager _rightManager;
    
    /** The form entry DAO */
    protected FormEntryDAO _formEntryDAO;

    /** The I18n utils */
    protected I18nUtils _i18nUtils;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
        _formEntryDAO = (FormEntryDAO) manager.lookup(FormEntryDAO.ROLE);
        _i18nUtils = (I18nUtils) manager.lookup(I18nUtils.ROLE);
    }
    
    /**
     * Generates statistics on each question of a form.
     * @param id The form id
     * @return A map containing the statistics
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> getStatistics(String id)
    {
        Form form = _resolver.resolveById(id);
        _formEntryDAO.checkHandleDataRight(form);
        
        return _getStatistics(form);
    }

    /**
     * Generate statistics of a mini-survey
     * No rights will be checked. Only a check that it is indeed a mini-survey will be done.
     * @param id the form id
     * @return a JSON map of the statistics
     */
    public Map<String, Object> getMiniSurveyStatistics(String id)
    {
        Form form = _resolver.resolveById(id);
        if (form.isMiniSurvey())
        {
            return _getStatistics(form);
        }
        else
        {
            return Map.of("error", "not-a-mini-survey");
        }
    }
    
    /**
     * Compute the statistics for each questions of the form
     * @param form the form
     * @return A JSON map representing the statistics
     */
    protected Map<String, Object> _getStatistics(Form form)
    {
        Map<String, Object> statistics = new HashMap<>();
        
        statistics.put("id", form.getId());
        statistics.put("title", form.getTitle());
        statistics.put("nbEntries", form.getEntries().size());
        statistics.put("questions", getStatsToArray(form));
        
        return statistics;
    }
    
    /**
     * Create a map with count of all answers per question
     * @param form current form
     * @return the statsMap
     */
    public Map<String, Map<String, Map<String, Object>>> getStatsMap(Form form)
    {
        Map<String, Map<String, Map<String, Object>>> statsMap = new LinkedHashMap<>();
        List<FormQuestion> questions = form.getQuestions()
                .stream()
                .filter(this::_displayField)
                .collect(Collectors.toList());
        
        List<FormEntry> entries = form.getEntries();
        
        for (FormQuestion question : questions)
        {
            Map<String, Map<String, Object>> questionValues = new LinkedHashMap<>();
            statsMap.put(question.getNameForForm(), questionValues);
            
            if (question.getType() instanceof MatrixQuestionType)
            {
                _dispatchMatrixStats(entries, question, questionValues);
            }
            else if (question.getType() instanceof ChoicesListQuestionType type)
            {
                if (type.getSourceType(question).remoteData())
                {
                    _dispatchChoicesWithRemoteDataStats(form, question, questionValues);
                }
                else
                {
                    _dispatchChoicesStats(form, question, questionValues);
                }
            }
            else if (question.getType() instanceof CheckBoxQuestionType)
            {
                _dispatchBooleanStats(form, entries, question, questionValues);
            }
            else
            {
                _dispatchStats(form, entries, question, questionValues);
            }
        }
        
        return statsMap;
    }
    
    /**
     * Transforms the statistics map into an array with some info.
     * @param form The form
     * @return A list of statistics.
     */
    public List<Map<String, Object>> getStatsToArray (Form form)
    {
        Map<String, Map<String, Map<String, Object>>> stats = getStatsMap(form);
        
        List<Map<String, Object>> result = new ArrayList<>();
        
        for (String questionNameForForm : stats.keySet())
        {
            Map<String, Object> questionMap = new HashMap<>();
            
            FormQuestion question = form.getQuestion(questionNameForForm);
            Map<String, Map<String, Object>> questionStats = stats.get(questionNameForForm);
            
            questionMap.put("id", questionNameForForm);
            questionMap.put("title", question.getTitle());
            questionMap.put("type", question.getType().getStorageType(question));
            questionMap.put("typeId", question.getType().getId());
            questionMap.put("mandatory", question.isMandatory());
            
            List<Object> options = new ArrayList<>();
            for (String optionId : questionStats.keySet())
            {
                Map<String, Object> option = new HashMap<>();

                option.put("id", optionId);

                if (question.getType() instanceof MatrixQuestionType)
                {
                    MatrixQuestionType type = (MatrixQuestionType) question.getType();
                    option.put("label", type.getRows(question).get(optionId));
                }
                
                questionStats.get(optionId).entrySet();
                List<Object> choices = new ArrayList<>();
                for (Entry<String, Object> choice : questionStats.get(optionId).entrySet())
                {
                    Map<String, Object> choiceMap = new HashMap<>();
                    
                    String choiceId = choice.getKey();
                    Object choiceOb = choice.getValue();
                    choiceMap.put("value", choiceId);
                    choiceMap.put("label", choiceOb instanceof Option ? ((Option) choiceOb).label() : choiceOb);

                    if (question.getType() instanceof MatrixQuestionType)
                    {
                        MatrixQuestionType type = (MatrixQuestionType) question.getType();
                        choiceMap.put("label", type.getColumns(question).get(choiceId));
                    }
                    
                    
                    choiceMap.put("count", choiceOb instanceof Option ? ((Option) choiceOb).count() : choiceOb);
                    
                    choices.add(choiceMap);
                }
                option.put("choices", choices);
                
                options.add(option);
            }
            questionMap.put("options", options);
            
            result.add(questionMap);
        }
        
        return result;
    }

    /**
     * Dispatch matrix stats
     * @param entries the entries
     * @param question the question
     * @param questionValues the values to fill
     */
    protected void _dispatchMatrixStats(List<FormEntry> entries, FormQuestion question, Map<String, Map<String, Object>> questionValues)
    {
        MatrixQuestionType matrixType = (MatrixQuestionType) question.getType();
        Map<String, String> rows = matrixType.getRows(question);
        if (rows != null)
        {
            for (String option : rows.keySet())
            {
                Map<String, Object> values = new LinkedHashMap<>();
                questionValues.put(option, values);
                
                Map<String, String> columns = matrixType.getColumns(question);
                if (columns != null)
                {
                    for (String column : columns.keySet())
                    {
                        values.put(column, 0);
                        _setOptionCount(question.getNameForForm(), entries, values, option, column);
                    }
                }
            }
        }
    }
    
    private void _setOptionCount(String questionId, List<FormEntry> entries, Map<String, Object> values, String rowValue, String columnValue)
    {
        int columnCount = (int) values.get(columnValue);
        for (FormEntry entry : entries)
        {
            Matrix matrix = entry.getValue(questionId);
            if (matrix != null)
            {
                List<String> options = matrix.get(rowValue);
                if (options != null && options.contains(columnValue))
                {
                    columnCount++;
                }
            }
        }
        values.put(columnValue, columnCount);
    }

    /**
     * Dispatch choices list stats
     * @param form the form
     * @param question the question
     * @param questionValues the values to fill
     */
    protected void _dispatchChoicesStats(Form form, FormQuestion question, Map<String, Map<String, Object>> questionValues)
    {
        Map<String, Object> values = new LinkedHashMap<>();
        questionValues.put("values", values);
        
        ChoicesListQuestionType type = (ChoicesListQuestionType) question.getType();
        ChoiceSourceType sourceType = type.getSourceType(question);
        Map<ChoiceOption, I18nizableText> options;
        try
        {
            Map<String, Object> enumParam = new HashMap<>();
            enumParam.put(AbstractSourceType.QUESTION_PARAM_KEY, question);
            options = sourceType.getTypedEntries(enumParam);
            
            for (ChoiceOption option : options.keySet())
            {
                String optionValue = (String) option.getValue();
                StringExpression expr = new StringExpression(question.getNameForForm(), Operator.EQ, optionValue);
                long countOption = _formEntryDAO.getFormEntries(form, false, expr, List.of()).size();
                Option choiceAttributes = new Option(options.get(option).getLabel(), countOption);
                values.put(optionValue, choiceAttributes);
            }
            
            if (type.hasOtherOption(question))
            {
                // Add other option
                Expression expr = new AndExpression(
                    new MetadataExpression(ChoicesListQuestionType.OTHER_PREFIX_DATA_NAME + question.getNameForForm()),
                    new NotExpression(new StringExpression(ChoicesListQuestionType.OTHER_PREFIX_DATA_NAME + question.getNameForForm(), Operator.EQ, StringUtils.EMPTY))
                );
                long countOtherOption = _formEntryDAO.getFormEntries(form, false, expr, List.of()).size();
                Option choiceAttributes = new Option(ChoicesListQuestionType.OTHER_OPTION_VALUE, countOtherOption);
                values.put(ChoicesListQuestionType.OTHER_OPTION_VALUE, choiceAttributes);
            }
            
            if (!type.isMandatory(question))
            {
                List<Expression> exprs = new ArrayList<>();
                
                exprs.add(new OrExpression(
                    new NotExpression(new MetadataExpression(question.getNameForForm())),
                    new StringExpression(question.getNameForForm(), Operator.EQ, StringUtils.EMPTY)
                ));
                
                if (type.hasOtherOption(question))
                {
                    exprs.add(new OrExpression(
                            new NotExpression(new MetadataExpression(ChoicesListQuestionType.OTHER_PREFIX_DATA_NAME + question.getNameForForm())),
                            new StringExpression(ChoicesListQuestionType.OTHER_PREFIX_DATA_NAME + question.getNameForForm(), Operator.EQ, StringUtils.EMPTY)
                        ));
                }
                
                long countOtherOption = _formEntryDAO.getFormEntries(form, false, new AndExpression(exprs), List.of()).size();
                Option choiceAttributes = new Option("__internal_not_answered", countOtherOption);
                values.put("_no_answer", choiceAttributes);
            }
        }
        catch (Exception e)
        {
            getLogger().error("An error occurred while trying to get choices options for question " + question.getId(), e);
        }
    }
    
    /**
     * Dispatch choices list with remote data stats
     * @param form the form
     * @param question the question
     * @param questionValues the values to fill
     */
    protected void _dispatchChoicesWithRemoteDataStats(Form form, FormQuestion question, Map<String, Map<String, Object>> questionValues)
    {
        Map<String, Object> values = new LinkedHashMap<>();
        questionValues.put("values", values);
        
        ChoicesListQuestionType type = (ChoicesListQuestionType) question.getType();
        ChoiceSourceType sourceType = type.getSourceType(question);

        try
        {
            long otherCount = 0;
            long noAnswer = 0;
            String nameForForm = question.getNameForForm();
            Map<Object, Long> stats = new LinkedHashMap<>();
            for (FormEntry entry : form.getEntries())
            {
                @SuppressWarnings("cast")
                List<Object> vals = entry.getValue(nameForForm) != null
                        ? entry.isMultiple(nameForForm)
                                ? Arrays.asList(entry.getValue(nameForForm))
                                : List.of((Object) entry.getValue(nameForForm)) // Need to cast in object because List.of want object and entry.getValue return typed value as string for exemple
                        : List.of();
                
                for (Object value : vals)
                {
                    Long count = stats.getOrDefault(value, 0L);
                    stats.put(value, count + 1);
                }

                if (type.hasOtherOption(question) && StringUtils.isNotBlank(entry.getValue(ChoicesListQuestionType.OTHER_PREFIX_DATA_NAME + nameForForm)))
                {
                    otherCount++;
                }
                else if (vals.isEmpty())
                {
                    noAnswer++;
                }
            }
        
            Map<String, Object> enumParam = new HashMap<>();
            enumParam.put(AbstractSourceType.QUESTION_PARAM_KEY, question);
            
            for (Object value : stats.keySet())
            {
                I18nizableText entry = sourceType.getEntry(new ChoiceOption(value), enumParam);
                if (entry != null)
                {
                    Option choiceAttributes = new Option(_i18nUtils.translate(entry), stats.get(value));
                    values.put(value.toString(), choiceAttributes);
                }
            }
            
            if (otherCount > 0)
            {
                Option choiceAttributes = new Option(ChoicesListQuestionType.OTHER_OPTION_VALUE, otherCount);
                values.put(ChoicesListQuestionType.OTHER_OPTION_VALUE, choiceAttributes);
            }
            
            if (noAnswer > 0)
            {
                Option choiceAttributes = new Option("__internal_not_answered", noAnswer);
                values.put("_no_answer", choiceAttributes);
            }
        }
        catch (Exception e)
        {
            getLogger().error("An error occurred while trying to get choices options for question " + question.getId(), e);
        }
    }
    
    /**
     * Dispatch boolean stats
     * @param form the form
     * @param entries the entries
     * @param question the question
     * @param questionValues the values to fill
     */
    protected void _dispatchBooleanStats(Form form, List<FormEntry> entries, FormQuestion question, Map<String, Map<String, Object>> questionValues)
    {
        Map<String, Object> values = new LinkedHashMap<>();
        questionValues.put("values", values);

        BooleanExpression booleanExpr = new BooleanExpression(question.getNameForForm(), true);
        int totalTrue = _formEntryDAO.getFormEntries(form, false, booleanExpr, List.of()).size();
        
        values.put("true", totalTrue);
        values.put("false", entries.size() - totalTrue);
    }
    
    /**
     * Dispatch default stats
     * @param form the form
     * @param entries the entries
     * @param question the question
     * @param questionValues the values to fill
     */
    protected void _dispatchStats(Form form, List<FormEntry> entries, FormQuestion question, Map<String, Map<String, Object>> questionValues)
    {
        Map<String, Object> values = new LinkedHashMap<>();
        questionValues.put("values", values);
        
        Expression expr = new AndExpression(
            new MetadataExpression(question.getNameForForm()),
            new NotExpression(new StringExpression(question.getNameForForm(), Operator.EQ, StringUtils.EMPTY))
        );
        
        int totalAnswered = _formEntryDAO.getFormEntries(form, false, expr, List.of()).size();
        values.put("answered", totalAnswered);
        values.put("empty", entries.size() - totalAnswered);
    }
    
    private boolean _displayField(FormQuestion question)
    {
        FormQuestionType type = question.getType();
        return type.canBeAnsweredByUser(question);
    }
    
    /**
     * Record representing a choice list option
     * @param label the label
     * @param count the number of time the option is selected
     */
    public record Option(String label, long count) { /* empty */ }
}
