/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.content.processing;

import java.io.File;
import java.time.LocalDate;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.chrono.IsoChronology;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.core.util.DateUtils;
import org.ametys.plugins.forms.content.Field;
import org.ametys.plugins.forms.content.data.FieldValue;
import org.ametys.plugins.forms.content.jcr.FormPropertiesManager;

/**
 * Abstract generator to sax form entry values
 *
 */
public abstract class AbstractFormFieldGenerator extends ServiceableGenerator
{
    /** The form properties manager */
    protected FormPropertiesManager _formPropManager;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _formPropManager = (FormPropertiesManager) smanager.lookup(FormPropertiesManager.ROLE);
    }
    
    /**
     * Sax a field value
     * @param fdValue the field value
     * @throws SAXException if an error occurred while saxing
     */
    protected void saxFieldValue(FieldValue fdValue) throws SAXException
    {
        saxFieldValue("field", fdValue);
    }
    
    /**
     * Sax a field value
     * @param tagName the root tag name
     * @param fdValue the field value
     * @throws SAXException if an error occurred while saxing
     */
    protected void saxFieldValue(String tagName, FieldValue fdValue) throws SAXException
    {
        Field field = fdValue.getField();
        Object rawValue = fdValue.getValue();
        
        String value = _getReadableValue(field, rawValue);
        
        if (value != null)
        {
            AttributesImpl atts = new AttributesImpl();
            atts.addCDATAAttribute("label", field.getLabel());
            atts.addCDATAAttribute("name", field.getName());
            atts.addCDATAAttribute("columnName", fdValue.getColumnName());
            atts.addCDATAAttribute("type", field.getType().name());
            
            Map<String, String> properties = field.getProperties();
            if (properties.containsKey("regexptype"))
            {
                atts.addCDATAAttribute("regexptype", properties.get("regexptype"));
            }
            
            XMLUtils.createElement(contentHandler, tagName, atts, value);
        }
    }
    
    /**
     * Get the readable value as String
     * @param field the field
     * @param rawValue the raw value. Cannot be null.
     * @return the value as String
     */
    protected String _getReadableValue(Field field, Object rawValue)
    {
        if (rawValue == null)
        {
            return null;
        }
        
        switch (field.getType())
        {
            case SELECT:
                boolean multiple = "true".equals(field.getProperties().get("multiple"));
                if (multiple)
                {
                    String[] rawValues = ((String) rawValue).split("\n");
                    List<String> readableValues = new ArrayList<>();
                    for (String v : rawValues)
                    {
                        readableValues.add(_formPropManager.getDisplayValue(field, v));
                    }
                    return StringUtils.join(readableValues, ", ");
                }
                else
                {
                    return _formPropManager.getDisplayValue(field, (String) rawValue);
                }
            case RADIO:
                String displayValue = _formPropManager.getDisplayValue(field, (String) rawValue);
                return StringUtils.isNotEmpty(displayValue) ? displayValue : (String) rawValue;
            case TEXT:
                String typedValue = StringUtils.trim((String) rawValue);
                String regexptype = field.getProperties().get("regexptype");
                if ("datetime".equals(regexptype) && StringUtils.isNotEmpty(typedValue))
                {
                    ZonedDateTime date = _getZonedDateTime(typedValue);
                    if (date == null)
                    {
                        LocalDate localDate = _getLocalDate(typedValue);
                        return localDate != null ? DateUtils.zonedDateTimeToString(localDate.atStartOfDay(ZoneOffset.UTC)) : null;
                    }
                    return DateUtils.zonedDateTimeToString(date); 
                }
                else if ("date".equals(regexptype) && StringUtils.isNotEmpty(typedValue))
                {
                    LocalDate localDate = _getLocalDate(typedValue);
                    if (localDate == null)
                    {
                        ZonedDateTime date = _getZonedDateTime(typedValue);
                        return date != null ? DateUtils.localDateToString(date.toLocalDate()) : null;
                    }
                    return DateUtils.localDateToString(localDate);
                }
                else
                {
                    // Trim value since some rendering (including default rendering) add many leading spaces
                    return typedValue;
                }
            case FILE:
                if (rawValue instanceof File)
                {
                    File file = (File) rawValue;
                    return file.getName();
                }
                else
                {
                    return String.valueOf(rawValue);
                }
            case PASSWORD:
            case CAPTCHA:
                // ignore value
                return null;
            default:
                return String.valueOf(rawValue);
        }
    }
    
    /**
     * Get the zoned date time from string value
     * @param value the string value
     * @return the zoned date time. Null if string value can't be parsed
     */
    protected ZonedDateTime _getZonedDateTime(String value)
    {
        try
        {
            DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd'T'HH:mm")
                    .withZone(ZoneId.systemDefault())
                    .withChronology(IsoChronology.INSTANCE);
            
            return ZonedDateTime.parse(value, formatter);
        }
        catch (Exception e) 
        {
            return null;
        }
    }
    
    /**
     * Get the local date from string value
     * @param value the string value
     * @return the local date time. Null if string value can't be parsed
     */
    protected LocalDate _getLocalDate(String value)
    {
        try
        {
            return DateUtils.parseLocalDate(value);
        }
        catch (Exception e) 
        {
            return null;
        }
    }
}
