/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.generators;

import java.io.IOException;
import java.time.LocalDate;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.core.right.RightManager;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.DateUtils;
import org.ametys.core.util.I18nUtils;
import org.ametys.plugins.forms.dao.FormDAO;
import org.ametys.plugins.forms.helper.LimitedEntriesHelper;
import org.ametys.plugins.forms.helper.ScheduleOpeningHelper;
import org.ametys.plugins.forms.question.FormQuestionType;
import org.ametys.plugins.forms.repository.Form;
import org.ametys.plugins.forms.repository.FormPage;
import org.ametys.plugins.forms.repository.FormPageRule;
import org.ametys.plugins.forms.repository.FormPageRule.PageRuleType;
import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.plugins.forms.repository.type.Rule;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.model.RepositoryDataContext;
import org.ametys.runtime.authentication.AccessDeniedException;

/**
 * SAX forms and its structures
 */
public class FormStructureGenerator extends ServiceableGenerator
{
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The current user provider */
    protected CurrentUserProvider _userProvider;
    
    /** I18n Utils */
    protected I18nUtils _i18nUtils;
    
    /** The limited entries helper */
    protected LimitedEntriesHelper _limitedEntriesHelper;
    
    /** The schedule opening helper */
    protected ScheduleOpeningHelper _scheduleOpeningHelper;

    /** The form DAO */
    protected FormDAO _formDAO;
    
    /** The right manager */
    protected RightManager _rightManager;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _userProvider = (CurrentUserProvider) serviceManager.lookup(CurrentUserProvider.ROLE);
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _i18nUtils = (I18nUtils) serviceManager.lookup(I18nUtils.ROLE);
        _limitedEntriesHelper = (LimitedEntriesHelper) serviceManager.lookup(LimitedEntriesHelper.ROLE);
        _scheduleOpeningHelper = (ScheduleOpeningHelper) serviceManager.lookup(ScheduleOpeningHelper.ROLE);
        _formDAO = (FormDAO) serviceManager.lookup(FormDAO.ROLE);
        _rightManager = (RightManager) serviceManager.lookup(RightManager.ROLE);
    }
    
    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        
        String id = parameters.getParameter("id", request.getParameter("id"));
        
        contentHandler.startDocument();
        
        UserIdentity user = _userProvider.getUser();
        Form form = _resolver.resolveById(id);
        boolean isPreview = StringUtils.equalsIgnoreCase(request.getParameter("preview"), "true");
        
        if (isPreview && !_rightManager.hasReadAccess(user, form))
        {
            throw new AccessDeniedException("User " + user.getLogin() + " tried to access form " + form.getId() + " without sufficient rights");
        }
        
        AttributesImpl attrs = new AttributesImpl();
        _addFormLimitationAttributes(request, form, attrs);
        attrs.addCDATAAttribute("id", form.getId());
        attrs.addCDATAAttribute("name", form.getName());
        attrs.addCDATAAttribute("isPreview", String.valueOf(isPreview));
        if (isPreview)
        {
            attrs.addCDATAAttribute("currentPageId", request.getParameter("currentPageId"));
            attrs.addCDATAAttribute("currentQuestionId", request.getParameter("currentQuestionId"));
        }
        
        if (form.hasWorkflow())
        {
            attrs.addCDATAAttribute("hasWorkflow", "true");
        }
        
        LocalDate startDate = form.getStartDate();
        LocalDate endDate = form.getEndDate();
        attrs.addCDATAAttribute("scheduleStatus", _scheduleOpeningHelper.getStatus(form).name());
        if (startDate != null)
        {
            attrs.addCDATAAttribute("startDate", DateUtils.localDateToString(startDate));
        }
        if (endDate != null)
        {
            attrs.addCDATAAttribute("endDate", DateUtils.localDateToString(endDate));
        }
        
        attrs.addCDATAAttribute("isConfigured", String.valueOf(_formDAO.isFormConfigured(form)));
        attrs.addCDATAAttribute("hasReadAccess", String.valueOf(_formDAO.hasReadRightOnForm(user, form)));
        attrs.addCDATAAttribute("canWrite", String.valueOf(_formDAO.hasWriteRightOnForm(user, form)));
        
        if (form.isMiniSurvey())
        {
            attrs.addCDATAAttribute("isMiniSurvey", "true");
        }
        
        XMLUtils.startElement(contentHandler, "form", attrs);
        XMLUtils.createElement(contentHandler, "title", form.getTitle());

        int pageNumber = 1;
        for (FormPage page : form.getPages())
        {
            if (!page.getQuestions().isEmpty())
            {
                saxPage(page, pageNumber);
                pageNumber++;
            }
        }
        
        XMLUtils.endElement(contentHandler, "form");
        
        contentHandler.endDocument();
    }

    /**
     * Add form limitation information to attributes
     * @param request the request
     * @param form the form
     * @param attrs the attributes to fill
     */
    protected void _addFormLimitationAttributes(Request request, Form form, AttributesImpl attrs)
    {
        UserIdentity currentUser = _userProvider.getUser();
        String clientIp = _limitedEntriesHelper.getClientIp(request);
        Optional<Long> maxEntriesOpt = form.getMaxEntries();
        boolean isClosed = false;
        attrs.addCDATAAttribute("isLimitedToOneEntryByUser", String.valueOf(form.isLimitedToOneEntryByUser()));
        attrs.addCDATAAttribute("hasEntries", String.valueOf(!form.getEntries().isEmpty()));
        if (form.isLimitedToOneEntryByUser() && _limitedEntriesHelper.hasUserAlreadyAnswer(form, currentUser, clientIp))
        {
            isClosed = true;
            attrs.addCDATAAttribute("user-already-answer", "true");
        }
        else if (form.isEntriesLimited() && maxEntriesOpt.isPresent())
        {
            Long maxEntries = maxEntriesOpt.get();
            int entriesCount = form.getActiveEntries().size();
            Optional<Long> queueSize = form.getQueueSize();
            // Form is open
            if (entriesCount < maxEntries)
            {
                String remainingMessage = form.getRemainingMessage().get();
                remainingMessage = remainingMessage.replace("{remaining}", String.valueOf(maxEntries - entriesCount));
                remainingMessage = remainingMessage.replace("{submissions}", String.valueOf(entriesCount));
                remainingMessage = remainingMessage.replace("{limit}", String.valueOf(maxEntries));
                attrs.addCDATAAttribute("limit-text", remainingMessage);
            }
            // Form is closed but there is an open queue
            else if (form.isQueueEnabled() && (queueSize.isEmpty() || entriesCount < (maxEntries + queueSize.get())))
            {
                String queueMessage = form.getClosedQueueMessage().get();
                queueMessage = queueMessage.replace("{inwaiting}", String.valueOf(entriesCount - maxEntries + 1));
                if (queueSize.isPresent())
                {
                    queueMessage = queueMessage.replace("{limit}", String.valueOf(queueSize.get()));
                }
                attrs.addCDATAAttribute("limit-text", queueMessage);
            }
            // Form is closed
            else
            {
                isClosed = true;
                String closedMessage = form.getClosedMessage().get();
                attrs.addCDATAAttribute("limit-text", closedMessage);
            }
        }
        
        attrs.addCDATAAttribute("isClosed", String.valueOf(isClosed));
    }
    
    /**
     * SAX a form page
     * @param page the page to SAX
     * @param pageNumber the page number
     * @throws SAXException if error occurs while SAXing
     */
    protected void saxPage (FormPage page, int pageNumber) throws SAXException
    {
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("id", page.getId());
        attrs.addCDATAAttribute("name", page.getName());
        attrs.addCDATAAttribute("number", String.valueOf(pageNumber));
        
        XMLUtils.startElement(contentHandler, "page", attrs);
        XMLUtils.createElement(contentHandler, "title", page.getTitle());
        
        for (FormQuestion question : page.getQuestions())
        {
            saxQuestion(question);
        }
        
        saxBranches(page);
        
        XMLUtils.endElement(contentHandler, "page");
    }
    
    /**
     * SAX a form question
     * @param question the question to SAX
     * @throws SAXException if error occurs while SAXing
     */
    protected void saxQuestion (FormQuestion question) throws SAXException
    {
        FormQuestionType type = question.getType();
        
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("canRead", String.valueOf(question.canRead(0L)));
        attrs.addCDATAAttribute("canWrite", String.valueOf(question.canWrite(0L)));
        
        attrs.addCDATAAttribute("type", type.getId());
        if (!type.onlyForDisplay(question))
        {
            attrs.addCDATAAttribute("storageType", type.getStorageType(question));
        }
        attrs.addCDATAAttribute("id", question.getId());
        attrs.addCDATAAttribute("name", question.getNameForForm());
        Optional<Rule> questionRule = question.getFirstQuestionRule();
        if (questionRule.isPresent())
        {
            Rule rule = questionRule.get();
            String action = rule.getAction().name();
            attrs.addCDATAAttribute("status", action);
        }
        attrs.addCDATAAttribute("onlyForDisplay", String.valueOf(type.onlyForDisplay(question)));
        attrs.addCDATAAttribute("canBeAnsweredByUser", String.valueOf(type.canBeAnsweredByUser(question)));
        XMLUtils.startElement(contentHandler, "question", attrs);
        
        RepositoryDataContext context = RepositoryDataContext.newInstance().withObject(question);
        question.dataToSAX(contentHandler, context);
        
        Map<String, Rule> questionsRules = question.getForm()
                .getQuestionsRule(question.getId())
                .entrySet()
                .stream()
                .collect(Collectors.toMap(
                    e -> e.getKey().getNameForForm(),
                    e -> e.getValue())
                );
        saxQuestionRules(questionsRules);
        
        XMLUtils.startElement(contentHandler, "additional-infos");
        type.saxAdditionalInfos(contentHandler, question);
        XMLUtils.endElement(contentHandler, "additional-infos");
        
        XMLUtils.endElement(contentHandler, "question");
    }
    
    /**
     * SAX branch logic
     * @param page the page
     * @throws SAXException if error occurs while SAXing
     */
    protected void saxBranches (FormPage page) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "branches");
        
        List<FormQuestion> questions = page.getQuestions();
        for (FormQuestion question : questions)
        {
            List<FormPageRule> rules = question.getPageRules();
            
            for (FormPageRule rule : rules)
            {
                AttributesImpl attr = new AttributesImpl();
                attr.addCDATAAttribute("id", question.getId());
                attr.addCDATAAttribute("name", question.getNameForForm());
                attr.addCDATAAttribute("value", rule.getOption());
                PageRuleType type = rule.getType();
                attr.addCDATAAttribute("type", type.name());
                if (type == PageRuleType.JUMP)
                {
                    attr.addCDATAAttribute("page", rule.getPageId());
                }
                XMLUtils.createElement(contentHandler, "rule", attr);
            }
        }
        
        FormPageRule rule = page.getRule();
        if (rule != null)
        {
            AttributesImpl attr = new AttributesImpl();
            PageRuleType type = rule.getType();
            attr.addCDATAAttribute("type", type.name());
            if (type == PageRuleType.JUMP)
            {
                attr.addCDATAAttribute("page", rule.getPageId());
            }
            XMLUtils.createElement(contentHandler, "page-rule", attr);
        }
        
        XMLUtils.endElement(contentHandler, "branches");
    }
    
    /**
     * SAX the rules related to this question
     * @param rules map of all the rules having current question as source grouped by target's nameForForm
     * @throws SAXException if error occurs while SAXing
     */
    protected void saxQuestionRules(Map<String, Rule> rules) throws SAXException
    {
        if (!rules.isEmpty())
        {
            XMLUtils.startElement(contentHandler, "rules");
            
            for (String targetNameForForm : rules.keySet())
            {
                AttributesImpl attr = new AttributesImpl();
                attr.addCDATAAttribute("target-name", targetNameForForm);
                attr.addCDATAAttribute("action", rules.get(targetNameForForm).getAction().name());
                attr.addCDATAAttribute("value", rules.get(targetNameForForm).getOption());
                XMLUtils.createElement(contentHandler, "rule", attr);
            }
            
            XMLUtils.endElement(contentHandler, "rules");
        }
    }
}
