/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * The forms DAO
 */
Ext.define('Ametys.plugins.forms.dao.FormDAO', {
    singleton: true,
    
    constructor: function()
    {
        /**
         * @callable
         * @member Ametys.plugins.forms.dao.FormDAO
         * @method getFormProperties 
         * Gets forms information.
         * This calls the method 'getFormProperties' of the server DAO 'FormDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {String[]} parameters.id The id of forms to retrieve
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object[]} callback.forms The form's properties. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object[]} callback.notAllowedForm The unauthorized forms
         * @param {String[]} callback.unknownForms The unknown forms
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.forms.dao.FormDAO",
                methodName: "getFormProperties",
                convertor: this._convertFormProperties,
                waitMessage: false,
                errorMessage: {
                    msg: "{{i18n DAOS_FORM_REQUEST_ERROR}}",
                    category: Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.forms.dao.FormDAO
         * @method createForm
         * Creates a form.
         * This calls the method 'createForm' of the server DAO 'FormDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {String} parameters.label The label
         * @param {String} parameters.title The title
         * @param {String} parameters.description The description
         * @param {String} parameters.endingMessage The ending message
         * @param {String} parameters.private A string representing a boolean indicating if the form is private
         * @param {String} parameters.siteName The site name
         * @param {String} parameters.language The language
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.forms.dao.FormDAO",
                methodName: "createForm",
                callback: {
                    handler: this._createFormCb
                },
                errorMessage: {
                    msg: "{{i18n PLUGINS_FORM_CREATE_FORM_ERROR}}",
                    category: Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.forms.dao.FormDAO
         * @method renameForm 
         * Edits a form.
         * This calls the method 'renameForm' of the server DAO 'FormDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {String} parameters.id The id of the form to edit
         * @param {String} parameters.label The label
         * @param {String} parameters.title The title
         * @param {String} parameters.description The description
         * @param {String} parameters.endingMessage The ending message
         * @param {String} parameters.private A string representing a boolean indicating if the form is private
         * @param {String} parameters.siteName The site name
         * @param {String} parameters.language The language
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.forms.dao.FormDAO",
                methodName: "renameForm",
                callback: {
                    handler: this._renameFormCb
                },
                errorMessage: {
                    msg: "{{i18n PLUGINS_FORM_RENAME_FORM_ERROR}}",
                    category: Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.forms.dao.FormDAO
         * @method copyForm 
         * Copies and pastes a form.
         * This calls the method 'copyForm' of the server DAO 'FormDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {String} parameters.formId The id of the form, target of the copy
         * @param {String} parameters.pageId The id of the page to copy
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.forms.dao.FormDAO",
                methodName: "copyForm",
                callback: {
                    handler: this._copyFormCb
                },
                errorMessage: {
                    msg: "{{i18n PLUGINS_FORMS_COPY_FORM_ERROR}}",
                    category: Ext.getClassName(this)
                }
            }
        ); 
        
        /**
         * @callable
         * @member Ametys.plugins.forms.dao.FormDAO
         * @method deleteForm 
         * Deletes a form.
         * This calls the method 'deleteForm' of the server DAO 'FormDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {String} parameters.id The id of the form to delete
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.forms.dao.FormDAO",
                methodName: "deleteForm",
                callback: {
                    handler: this._deleteFormCb
                },
                errorMessage: {
                    msg: "{{i18n PLUGINS_FORMS_DELETE_FORM_ERROR}}",
                    category: Ext.getClassName(this)
                },
                localParamsIndex: 1
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.forms.dao.FormDAO
         * @method moveForm 
         * Moves a form
         * This calls the method 'moveForm' of the server DAO 'FormDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {String} parameters.id The id of the form
         * @param {String} parameters.newParentId The id of the new parent directory of the form
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.forms.dao.FormDAO",
                methodName: "moveForm",
                callback: {
                    handler: this._moveFormCb
                },
                errorMessage: {
                    msg: "{{i18n PLUGINS_FORMS_MOVE_FAILED}}",
                    category:Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.forms.dao.FormDAO
         * @method getStatistics 
         * Generates statistics on each question of a form.
         * This calls the method 'getStatistics' of the server DAO 'FormDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {String} parameters.id The id of the form.
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.forms.helper.FormStatisticsHelper",
                methodName: "getStatistics",
                callback: {
                    handler: Ext.emptyFn
                },
                errorMessage: {
                    msg: "{{i18n PLUGINS_FORMS_STATISTICS_TOOL_QUERY_ERROR}}",
                    category: Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.forms.dao.FormDAO
         * @method setWorkflow 
         * Set a new workflow to a form.
         * This calls the method 'setWorkflow' of the server DAO 'FormDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {String} parameters.id The id of the form to edit
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.forms.dao.FormDAO",
                methodName: "setWorkflow",
                callback: {
                    handler: this._setWorkflowCb
                },
                errorMessage: {
                    msg: "{{i18n PLUGINS_FORMS_SET_WORKFLOW_FORM_ERROR}}",
                    category: Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.forms.dao.FormDAO
         * @method getFormTitle 
         * Get the form title.
         * This calls the method 'getFormTitle' of the server DAO 'FormDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {String} parameters.id The id of the form to edit
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.forms.dao.FormDAO",
                methodName: "getFormTitle",
                callback: {
                    handler: Ext.emptyFn
                },
                errorMessage: {
                    msg: "{{i18n DAOS_FORM_REQUEST_ERROR}}",
                    category: Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.forms.dao.FormDAO
         * @method getFormFullPath 
         * Set a new workflow to a form.
         * This calls the method 'getFormFullPath' of the server DAO 'FormDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {String} parameters.id The id of the form to edit
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.forms.dao.FormDAO",
                methodName: "getFormFullPath",
                callback: {
                    handler: Ext.emptyFn
                },
                errorMessage: {
                    msg: "{{i18n DAOS_FORM_REQUEST_ERROR}}",
                    category: Ext.getClassName(this)
                }
            }
        );
    },
    
    /**
     * Callback function called after {@link #createForm} has been processed.
     * @param {Object} response The text response provided by the {@link Ametys.data.ServerComm}
     * @private
     */
    _createFormCb: function(response)
    {
        Ext.create("Ametys.message.Message", {
            type: Ametys.message.Message.CREATED,
            targets: {
                id: Ametys.message.MessageTarget.FORM_TARGET,
                parameters: {
                    id: response.id
                }
            }
        });
        
        Ext.create("Ametys.message.Message", {
            type: Ametys.message.Message.CREATED,
            targets: {
                id: Ametys.message.MessageTarget.PAGE,
                parameters: {
                    id: response.id
                }
            }
        });
    },
    
    /**
     * Callback function called after {@link #renameForm} has been processed.
     * @param {Object} response The text response provided by the {@link Ametys.data.ServerComm}
     * @private
     */
    _renameFormCb: function(response)
    {
        Ext.create("Ametys.message.Message", {
            type: Ametys.message.Message.MODIFIED,
            targets: {
                id: Ametys.message.MessageTarget.FORM_TARGET,
                parameters: {
                    id: response.id
                }
            }
        });
    },
    
    /**
     * Callback function called after {@link #deleteForm} has been processed.
     * @param {Object} response The text response provided by the {@link Ametys.data.ServerComm}
     * @param {Object} args The callback arguments
     * @param {Object[]} params The callback parameters (server-side and client-side)
     * @private
     */
    _deleteFormCb: function(response, args, params)
    {
        var formTool = Ametys.tool.ToolsManager.getTool("uitool-form");
        if(formTool && response.id == formTool.getParams().id)
        {
            formTool.close();
        }
        
        var entriesTool = Ametys.tool.ToolsManager.getTool("uitool-form-entries$" + response.id);
        if(entriesTool)
        {
            entriesTool.close();
        }
        
        Ext.create("Ametys.message.Message", {
            type: Ametys.message.Message.DELETED,
            targets: params[1] // The deleted form target
        });
     },
     
     /**
     * Callback function called after {@link #copyForm} has been processed.
     * @param {Object} response The text response provided by the {@link Ametys.data.ServerComm}
     * @private
     */
    _copyFormCb: function(response)
    {
        Ext.create("Ametys.message.Message", {
            type: Ametys.message.Message.CREATED,
            targets: {
                id: Ametys.message.MessageTarget.FORM_TARGET,
                parameters: {
                    id: response.id
                }
            }
        });
    },
     
     /**
     * Callback function called after {@link #moveForm} has been processed.
     * @param {Object} response The server response
     * @private
     */
    _moveFormCb: function(response)
    {
        if (response.message == 'not-allowed')
        {
            Ametys.Msg.show({
                title: "{{i18n PLUGINS_FORMS_NOT_ALLOWED_TITLE}}",
                msg: "{{i18n PLUGINS_FORMS_NOT_ALLOWED_MSG}}",
                buttons: Ext.Msg.OK,
                icon: Ext.Msg.ERROR
            });
        }
        else if (response.message == 'cannot-move')
        {
            Ametys.Msg.show({
                title: "{{i18n PLUGINS_FORMS_CANNOT_MOVE_TITLE}}",
                msg: "{{i18n PLUGINS_FORMS_CANNOT_MOVE_MSG}}",
                buttons: Ext.Msg.OK,
                icon: Ext.Msg.WARNING
            });
        }
        else
        {   
            var formId = response.id;

            Ext.create("Ametys.message.Message", {
                type: Ametys.message.Message.MOVED,
                targets: {
                    id: Ametys.message.MessageTarget.FORM_TARGET,
                    parameters: {
                        id: formId
                    }
                }
            });
        }
    },
    
    /**
     * Callback function called after {@link #setWorkflow} has been processed.
     * @param {Object} response The text response provided by the {@link Ametys.data.ServerComm}
     * @private
     */
    _setWorkflowCb: function(response)
    {
        Ext.create("Ametys.message.Message", {
            type: Ametys.message.Message.MODIFIED,
            targets: {
                id: Ametys.message.MessageTarget.FORM_TARGET,
                parameters: {
                    id: response.id
                }
            }
        });
    },
    
    /**
     * @private
     * Convert function called after #getFormProperties is processed
     * @param {Object} formProperties The server response
     */
    _convertFormProperties: function (formProperties)
    {
        return Ext.create ('Ametys.plugins.forms.Form', formProperties);
    },
});