/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.generators;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.workflow.AmetysObjectCheckRightsCondition;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.DateUtils;
import org.ametys.plugins.forms.FormsException;
import org.ametys.plugins.forms.dao.FormDAO;
import org.ametys.plugins.forms.dao.FormEntryDAO;
import org.ametys.plugins.forms.data.Answer;
import org.ametys.plugins.forms.repository.Form;
import org.ametys.plugins.forms.repository.FormEntry;
import org.ametys.plugins.forms.workflow.AbstractFormDashboardGenerator;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.UserExpression;
import org.ametys.plugins.workflow.support.WorkflowProvider;
import org.ametys.runtime.i18n.I18nizableText;

import com.opensymphony.workflow.Workflow;
import com.opensymphony.workflow.loader.ActionDescriptor;
import com.opensymphony.workflow.loader.WorkflowDescriptor;

/**
 * This class generates all the forms process information for current user
 */
public class FormDashboardGenerator extends AbstractFormDashboardGenerator
{
    /** The workflow provider */
    protected WorkflowProvider _workflowProvider;
    
    /** The form DAO */
    protected FormDAO _formDAO;
    
    /** The form entry DAO */
    protected FormEntryDAO _formEntryDAO;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _workflowProvider = (WorkflowProvider) smanager.lookup(WorkflowProvider.ROLE);
        _formDAO = (FormDAO) smanager.lookup(FormDAO.ROLE);
        _formEntryDAO = (FormEntryDAO) smanager.lookup(FormEntryDAO.ROLE);
    }
    
    @Override
    protected List<Answer> _getAnswers(Request request, String siteName, String lang, UserIdentity user) throws FormsException
    {
        UserExpression userExpression = new UserExpression(FormEntry.ATTRIBUTE_USER, Operator.EQ, user);
        
        List<Answer> collect = _formDAO.getForms(siteName)
            .stream()
            .map(f -> _formEntryDAO.getFormEntries(f, false, userExpression, List.of()))
            .flatMap(List::stream)
            .map(this::_formEntry2Answer)
            .collect(Collectors.toList());
        
        return collect;
    }
    
    /**
     * Transform form entry to answer object
     * @param entry the form entry
     * @return the answer object
     */
    protected Answer _formEntry2Answer(FormEntry entry)
    {
        Form form = entry.getForm();
        String workflowName = form.getWorkflowName();
        return new Answer(
                entry.getId(),
                entry.getEntryId(),
                form.getId(), 
                form.getTitle(), 
                DateUtils.asDate(entry.getSubmitDate()), 
                form.getWorkflowName(),
                StringUtils.isNotBlank(workflowName) ? (int) entry.getWorkflowId() : -1,
                entry.getUser()
            );
    }
    
    @Override
    protected Workflow _getWorkflow(Answer answer)
    {
        FormEntry formAnswer = _resolver.resolveById(answer.getId());
        return _workflowProvider.getAmetysObjectWorkflow(formAnswer);
    }
    
    @Override
    protected void _saxWorkflowInformations(Answer answer, UserIdentity user) throws Exception
    {
        super._saxWorkflowInformations(answer, user);
        
        String workflowName = answer.getWorkflowName();
        if (StringUtils.isNotBlank(workflowName))
        {
            FormEntry formEntry = _resolver.resolveById(answer.getId());
            
            Workflow workflow = _getWorkflow(answer);
            WorkflowDescriptor workflowDescriptor = _workflowHelper.getWorkflowDescriptor(answer.getWorkflowName());
            
            Map<String, Object> inputs = new HashMap<>();
            inputs.put(AmetysObjectCheckRightsCondition.AMETYS_OBJECT_KEY, formEntry);
            
            int[] availableActions = workflow.getAvailableActions(answer.getWorkflowId(), inputs);
            for (int actionId : availableActions)
            {
                ActionDescriptor action = workflowDescriptor.getAction(actionId);
                
                AttributesImpl attrs = new AttributesImpl();
                attrs.addCDATAAttribute("id", String.valueOf(actionId));
                
                Map<String, Object> metaAttributes = action.getMetaAttributes();
                String actionStatus = (String) metaAttributes.getOrDefault("status", String.valueOf(actionId));
                attrs.addCDATAAttribute("status", actionStatus);
                
                _saxActionType(action, attrs);
                
                I18nizableText workflowActionLabel = new I18nizableText("application", action.getName());
                
                XMLUtils.startElement(contentHandler, "action", attrs);
                workflowActionLabel.toSAX(contentHandler);
                XMLUtils.endElement(contentHandler, "action");
                
            }
        }
    }
    
    private void _saxActionType(ActionDescriptor action, AttributesImpl attrs)
    {
        Map<String, Object> metaAttributes = action.getMetaAttributes();
        
        String actionType = (String) metaAttributes.get("action-type");
        if (actionType != null)
        {
            attrs.addCDATAAttribute("type", actionType);
        }
    }
}
