/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.helper;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.workflow.AmetysObjectCheckRightsCondition;
import org.ametys.core.right.RightManager;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.forms.dao.FormDAO;
import org.ametys.plugins.forms.dao.FormEntryDAO;
import org.ametys.plugins.forms.repository.Form;
import org.ametys.plugins.forms.repository.FormEntry;
import org.ametys.plugins.workflow.support.WorkflowHelper;
import org.ametys.plugins.workflow.support.WorkflowProvider;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

import com.opensymphony.workflow.Workflow;
import com.opensymphony.workflow.loader.ActionDescriptor;
import com.opensymphony.workflow.loader.WorkflowDescriptor;

/**
 * The helper to handle admin dashboard
 */
public class FormAdminDashboardHelper extends AbstractLogEnabled implements Serviceable, Component
{
    /** Avalon ROLE. */
    public static final String ROLE = FormAdminDashboardHelper.class.getName();
    
    /** The workflow provider */
    protected WorkflowProvider _workflowProvider;
    
    /** The form DAO */
    protected FormDAO _formDAO;
    
    /** The right manager */
    protected RightManager _rightManager;
    
    /** The form entry DAO */
    protected FormEntryDAO _formEntryDAO;
    
    /** The workflow helper component */
    protected WorkflowHelper _workflowHelper;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _workflowProvider = (WorkflowProvider) manager.lookup(WorkflowProvider.ROLE);
        _formDAO = (FormDAO) manager.lookup(FormDAO.ROLE);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
        _formEntryDAO = (FormEntryDAO) manager.lookup(FormEntryDAO.ROLE);
        _workflowHelper = (WorkflowHelper) manager.lookup(WorkflowHelper.ROLE);
    }
    
    /**
     * The list of form to admin for current user
     * @param siteName the sitename
     * @param currentUser the current user
     * @return the list of form to admin for current user
     */
    public List<Form> getFormToAdmin(String siteName, UserIdentity currentUser)
    {
        return _formDAO.getForms(siteName)
                .stream()
                .filter(Form::hasWorkflow)
                .filter(f -> _formEntryDAO.hasHandleDataRightOnForm(currentUser, f))
                .filter(f -> this._hasAtLeastOneEntryToDoAction(f, currentUser))
                .toList();
    }
    
    /**
     * <code>true</code> if the form has an entry that the user can do action
     * @param form the form
     * @param user the user
     * @return <code>true</code> if the form has an entry that the user can do action
     */
    protected boolean _hasAtLeastOneEntryToDoAction(Form form, UserIdentity user)
    {
        return form.getEntries()
                .stream()
                // filter entry that have been anonymized
                .filter(entry -> !entry.hasValue(FormEntry.ATTRIBUTE_ANONYMIZATION_DATE))
                .anyMatch(this::hasAvailableActions);
    }
    
    /**
     * <code>true</code> if the entry has available actions for current user
     * @param entry the entry
     * @return <code>true</code> if the entry has available actions for current user
     */
    public boolean hasAvailableActions(FormEntry entry)
    {
        return _getAvailableActions(entry).length > 0;
    }
    
    /**
     * Get available actions for current user filter by its type
     * @param entry the entry
     * @param types the types of actions. Empty to get all available actions
     * @return the available actions for current user
     */
    public List<ActionDescriptor> getAvailableActions(FormEntry entry, List<String> types)
    {
        List<ActionDescriptor> actions = new ArrayList<>();
        
        WorkflowDescriptor workflowDescriptor = _workflowHelper.getWorkflowDescriptor(entry.getForm().getWorkflowName());
        for (int actionId : _getAvailableActions(entry))
        {
            ActionDescriptor action = workflowDescriptor.getAction(actionId);
            Map<String, Object> metaAttributes = action.getMetaAttributes();
            String actionType = (String) metaAttributes.getOrDefault("action-type", StringUtils.EMPTY);
            if (types.isEmpty() || types.contains(actionType))
            {
                actions.add(action);
            }
            
        }
        
        return actions;
    }
    
    private int[] _getAvailableActions(FormEntry entry)
    {
        Map<String, Object> inputs = new HashMap<>();
        inputs.put(AmetysObjectCheckRightsCondition.AMETYS_OBJECT_KEY, entry);
     
        Workflow workflow = _workflowProvider.getAmetysObjectWorkflow(entry);
        return workflow.getAvailableActions(entry.getWorkflowId(), inputs);
    }
}
