/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

const NameHelper = Java.type('org.ametys.plugins.repository.jcr.NameHelper');

migrateFormDirectories(Repository.session);
migrateFormDirectories(Repository.liveSession);
migrateForms(Repository.session);
migrateForms(Repository.liveSession);
moveFormsPluginNode(Repository.session);
moveFormsPluginNode(Repository.liveSession);


function migrateFormDirectories(s)
{
    let migratedDirectories = 0;

    try
    {
        Repository.query("//element(*, ametys:forms)", true, s).forEach(function(formDirectory)
        {
            let name = formDirectory.getName();
            let node = formDirectory.getNode();
            
            if (!node.hasProperty("ametys:title") && name != "ametys:forms")
            {
                // set title
                formDirectory.setTitle(name);
                renameAndMoveIfNeeded("formdirectory-", name, formDirectory)
                migratedDirectories++;
            }
        })
    }
    catch (e)
    {
        logger.error("An error has occured during migration of form directories", e);
        throw e;
    } 
    
    if (migratedDirectories > 0)
    {
        s.save();
    }
    
    logger.info(`${migratedDirectories} form directories have been migrated in ${s.getWorkspace().getName()} workspace`);
}

function migrateForms(s)
{
    let migratedForms = 0;

    try
    {
        Repository.query("//element(*, ametys:form)", true, s).forEach(function(form)
        {
            renameAndMoveIfNeeded("form-", form.getTitle(), form)
            migratedForms++;
        })
    }
    catch (e)
    {
        logger.error("An error has occured during migration of forms", e);
        throw e;
    } 
    
    if (migratedForms > 0)
    {
        s.save();
    }
    
    logger.info(`${migratedForms} forms have been migrated in ${s.getWorkspace().getName()} workspace`);
}

function moveFormsPluginNode(s)
{
    let count = 0;
    
    try
    {
        // rename 'ametys-internal:plugins/formsdirectory' node to 'ametys-internal:plugins/forms'
        const nodes = Repository.query("//element(formsdirectory, ametys:unstructured)", false, s)
        while (nodes.hasNext())
        {
            const node = nodes.next();
            if (node.getParent().getName() == 'ametys-internal:plugins')
            {
                node.getSession().move(node.getPath(), node.getParent().getPath() + '/forms');
                count++;
            }
        }
    }
    catch (e)
    {
        logger.error("An error has occured during renaming of forms' plugin root node", e);
        throw e;
    } 
    
    if (count > 0)
    {
        s.save();
    }
    
    logger.info(`${count} forms' plugin root nodes has been renamed in ${s.getWorkspace().getName()} workspace`);  
}

function renameAndMoveIfNeeded(prefix, name, ametysObject)
{
    let node = ametysObject.getNode();
    let legalName;
    try
    {
        legalName = NameHelper.filterName(name);
    }
    catch(e)
    {
        legalName = NameHelper.filterName(prefix + name);
    }

    if (legalName != name)
    {
        let parentDirectory = ametysObject.getParent();
      
        // Find unique name from legal name
        var uniqueName = legalName;
        var index = 2;
        while (parentDirectory.hasChild(uniqueName))
        {
            uniqueName = legalName + "-" + (index++);
        }
        
        // rename jcr node
        node.getSession().move(node.getPath(), node.getParent().getPath() + '/' + uniqueName);
    }
}
