/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms;

import javax.jcr.ItemExistsException;
import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.Session;

import org.ametys.plugins.forms.dao.FormDirectoryDAO;
import org.ametys.plugins.forms.repository.Form;
import org.ametys.plugins.forms.repository.FormDirectory;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.MovableAmetysObject;
import org.ametys.plugins.repository.RepositoryIntegrityViolationException;
import org.ametys.plugins.repository.jcr.SimpleAmetysObject;

/**
 * Common methods for {@link Form} and {@link FormDirectory}
 */
public final class FormAndDirectoryCommonMethods
{
    private FormAndDirectoryCommonMethods()
    {
    }
    
    /**
     * Implementation of {@link MovableAmetysObject#canMoveTo(AmetysObject)}
     * @param siteName Name of current site
     * @param newParent See moveTo.
     * @param formOrFormDirectory The object to move
     * @param formDAO The Form DAO
     * @return true if the move operation may succeed. If false is returned and you call moveTo anyway, you may encontered a RuntimeException (such as UnsupportedOperationException)
     * @throws AmetysRepositoryException if an error occurs.
     */
    public static boolean canMoveTo(String siteName, AmetysObject newParent, SimpleAmetysObject formOrFormDirectory, FormDirectoryDAO formDAO) throws AmetysRepositoryException
    {
        return newParent instanceof FormDirectory
                && !formOrFormDirectory.equals(formDAO.getFormDirectoriesRootNode(siteName));
    }
    
    /**
     * Implementation of {@link MovableAmetysObject#moveTo(AmetysObject, boolean)}
     * @param newParent The new parent for the current object. Can not be null. Can not be a child of the current node. Must be a TraversableAmetyObject.
     * @param renameIfExist true to rename moved page if a page with same name already exist
     * @param formOrFormDirectory The object to move
     * @throws AmetysRepositoryException if an error occurs.
     * @throws RepositoryIntegrityViolationException if a page with the same name already exists.
     */
    public static void moveTo(AmetysObject newParent, boolean renameIfExist, SimpleAmetysObject formOrFormDirectory) throws AmetysRepositoryException, RepositoryIntegrityViolationException
    {
        Node srcNode = formOrFormDirectory.getNode();
        Node targetNode = ((FormDirectory) newParent).getNode(); // assume canMoveTo was called, so cast can be done
        
        try
        {
            if (formOrFormDirectory.getParent().equals(newParent))
            {
                // Do nothing
            }
            else
            {
                String name = srcNode.getName();
                if (renameIfExist)
                {
                    int index = 1;
                    String origName = name;
                    while (targetNode.hasNode(name))
                    {
                        name = origName + "-" + index++;
                    }
                }
                String newPath = targetNode.getPath() + "/" + name;
                
                Session session = srcNode.getSession();
                try
                {
                    session.move(srcNode.getPath(), newPath);
                }
                catch (ItemExistsException e)
                {
                    throw new AmetysRepositoryException("A form already exists for new path '" + newPath + "'", e);
                }
                session.save();
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to move form '" + formOrFormDirectory + "' to node '" + newParent.getId() + "'", e);
        }
    }
    
    /**
     * Implementation of {@link MovableAmetysObject#orderBefore(AmetysObject)}
     * @param siblingNode The node that will be the next sibling node of the current node. Must have the same parent as the current node. Can be null to set the current node as the last node.
     * @param formOrFormDirectory The object to move
     * @throws AmetysRepositoryException if an error occurs.
     */
    public static void orderBefore(AmetysObject siblingNode, SimpleAmetysObject formOrFormDirectory) throws AmetysRepositoryException
    {
        throw new UnsupportedOperationException("Form ordering is not supported");
    }
}
