/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.content.data;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.repository.Content;
import org.ametys.core.cocoon.JSonReader;
import org.ametys.core.right.RightManager;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.DateUtils;
import org.ametys.core.util.I18nUtils;
import org.ametys.plugins.core.user.UserHelper;
import org.ametys.plugins.forms.FormsException;
import org.ametys.plugins.forms.content.Field;
import org.ametys.plugins.forms.content.Field.FieldType;
import org.ametys.plugins.forms.content.Form;
import org.ametys.plugins.forms.content.jcr.FormPropertiesManager;
import org.ametys.plugins.forms.content.table.FormTableManager;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.workflow.store.JdbcWorkflowStore;
import org.ametys.plugins.workflow.support.WorkflowProvider;
import org.ametys.runtime.i18n.I18nizableText;

import com.opensymphony.workflow.Workflow;
import com.opensymphony.workflow.loader.StepDescriptor;
import com.opensymphony.workflow.loader.WorkflowDescriptor;
import com.opensymphony.workflow.spi.Step;

/**
 * Get the submitted entries of a form
 *
 */
public class GetFormEntriesAction extends ServiceableAction
{
    /** Pattern for options value */
    protected static final Pattern __OPTION_VALUE_PATTERN = Pattern.compile("^option-([0-9]+)-value$");
            
    /** The internationalizable text symbolizing the absence of workflow step */
    protected static final I18nizableText __MESSAGE_NO_STEP = new I18nizableText("plugin.forms", "PLUGINS_FORMS_UITOOL_ENTRY_WORKFLOW_NO_WORKFLOW");
    
    /** The form properties manager. */
    protected FormPropertiesManager _formPropertiesManager;
    
    /** The form data manager. */
    protected FormTableManager _formTableManager;
    
    /** The ametys object resolver. */
    protected AmetysObjectResolver _resolver;
    
    /** The workflow provider */
    protected WorkflowProvider _workflowProvider;
    
    /** Utility component for internationalizable text */
    protected I18nUtils _i18nUtils;
    
    /** The user helper */
    protected UserHelper _userHelper;
    
    /** The right manager */
    protected RightManager _rightManager;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _formPropertiesManager = (FormPropertiesManager) smanager.lookup(FormPropertiesManager.ROLE);
        _formTableManager = (FormTableManager) smanager.lookup(FormTableManager.ROLE);
        _workflowProvider = (WorkflowProvider) smanager.lookup(WorkflowProvider.ROLE);
        _i18nUtils = (I18nUtils) smanager.lookup(I18nUtils.ROLE);
        _userHelper = (UserHelper) smanager.lookup(UserHelper.ROLE);
        _rightManager = (RightManager) smanager.lookup(RightManager.ROLE);
    }
    
    @Override
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        @SuppressWarnings("unchecked")
        Map<String, Object> jsParameters = (Map<String, Object>) objectModel.get(ObjectModelHelper.PARENT_CONTEXT);
        
        String siteName = (String) jsParameters.get("siteName");
        String contentId = (String) jsParameters.get("contentId");
        String formId = (String) jsParameters.get("id");
        
        int start = jsParameters.containsKey("start") ? (int) jsParameters.get("start") : 0;
        int limit = jsParameters.containsKey("limit") ? (int) jsParameters.get("limit") : Integer.MAX_VALUE;
        
        Map<String, Object> result = new HashMap<>();
        List<Map<String, Object>> entries2json = new ArrayList<>();
        
        Content content = _resolver.resolveById(contentId);
        if (_rightManager.currentUserHasReadAccess(content))
        {
            Form form = _formPropertiesManager.getForm(siteName, formId);
            
            if (form == null)
            {
                throw new ProcessingException("The form of ID '" + formId + " can't be found in the site '" + siteName + "'.");
            }
            
            Workflow workflow = _workflowProvider.getExternalWorkflow(JdbcWorkflowStore.ROLE);
            
            try
            {
                Map<String, FieldValue> columns = _formTableManager.getColumns(form);
                List<UserEntry> entries = _formTableManager.getSubmissions(form, columns, start, limit, null);
                int totalSubmissions = _formTableManager.getTotalSubmissions(form.getId());
                
                result.put("id", form.getId());
                result.put("label", form.getLabel());
                result.put("total", totalSubmissions);
                
                for (UserEntry entry : entries)
                {
                    entries2json.add(_entry2json(entry, workflow));
                }
            }
            catch (FormsException e)
            {
                getLogger().error("Failed to get SQL table for form '" + form.getId() + "' for content of id.", e);
            }
        }
        
        result.put("entries", entries2json);
        
        Request request = ObjectModelHelper.getRequest(objectModel);
        request.setAttribute(JSonReader.OBJECT_TO_READ, result);
        return EMPTY_MAP;
    }
    
    private Map<String, Object> _entry2json(UserEntry entry, Workflow workflow)
    {
        Map<String, Object> entryAsMap = new HashMap<>();
        
        entryAsMap.put("id", entry.getId());
        entryAsMap.put("submission-date", DateUtils.dateToString(entry.getCreationDate()));
        
        if (entry.getWorkflowId() != null)
        {
            entryAsMap.put("workflowStep", _workflow2json (entry, workflow));
        }
        
        UserIdentity userIdentity = entry.getUserIdentity();
        if (userIdentity != null)
        {
            entryAsMap.put("user", _userHelper.user2json(userIdentity, true));
        }
        
        for (FieldValue fdValue : entry.getValues())
        {
            Object rawValue = fdValue.getValue();
            if (rawValue != null)
            {
                entryAsMap.put(fdValue.getColumnName() + "_raw", rawValue);
                entryAsMap.put(fdValue.getColumnName(), _getReadableValue(fdValue.getField(), rawValue));
            }
        }
        
        return entryAsMap;
    }
    
    private Map<String, Object> _workflow2json (UserEntry entry, Workflow workflow)
    {
        Map<String, Object> workflowInfos = new LinkedHashMap<>();
        
        int currentStepId = 0;
        
        long workflowId = entry.getWorkflowId();
        Iterator<Step> currentSteps = workflow.getCurrentSteps(workflowId).iterator();
        
        while (currentSteps.hasNext())
        {
            Step step = currentSteps.next();
            currentStepId = step.getStepId();
        }
        
        String workflowName = workflow.getWorkflowName(workflowId);
        WorkflowDescriptor workflowDescriptor = workflow.getWorkflowDescriptor(workflowName);
        StepDescriptor stepDescriptor = workflowDescriptor.getStep(currentStepId);
        
        if (stepDescriptor != null)
        {
            I18nizableText workflowStepName = new I18nizableText("application", stepDescriptor.getName());
            
            workflowInfos.put("stepId", currentStepId);
            workflowInfos.put("name", workflowStepName);
            
            String[] icons = new String[] {"small", "medium", "large"};
            for (String icon : icons)
            {
                if ("application".equals(workflowStepName.getCatalogue()))
                {
                    workflowInfos.put(icon + "Icon", "/plugins/cms/resources_workflow/" + workflowStepName.getKey() + "-" + icon + ".png");
                }
                else
                {
                    String pluginName = workflowStepName.getCatalogue().substring("plugin.".length());
                    workflowInfos.put(icon + "Icon", "/plugins/" + pluginName + "/resources/img/workflow/" + workflowStepName.getKey() + "-" + icon + ".png");
                }
            }
        }
        
        return workflowInfos;
    }
    
    /**
     * Get label to display for field
     * @param field The field
     * @param value The value
     * @return The value to display
     */
    protected String _getReadableValue (Field field, Object value)
    {
        if (field.getType().equals(FieldType.SELECT) || field.getType().equals(FieldType.RADIO))
        {
            Map<String, String> properties = field.getProperties();
            for (String key : properties.keySet())
            {
                Matcher matcher = __OPTION_VALUE_PATTERN.matcher(key);
                if (matcher.matches())
                {
                    if (StringUtils.trim((String) value).equals(StringUtils.trim(properties.get(key))))
                    {
                        String index = matcher.group(1);
                        String label = properties.containsKey("option-" + index + "-label") ? properties.get("option-" + index + "-label") : null;
                        if (StringUtils.isNotEmpty(label))
                        {
                            return label;
                        }
                    }
                }
            }
        }
        
        return Optional.ofNullable(value)
                       .map(Object::toString)
                       .orElse(null);
    }
}
