/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.generators;

import java.util.ArrayList;
import java.util.List;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.SAXException;

import org.ametys.core.right.RightManager;
import org.ametys.plugins.forms.dao.FormDAO;
import org.ametys.plugins.forms.helper.FormAdminDashboardHelper;
import org.ametys.plugins.forms.repository.Form;
import org.ametys.plugins.linkdirectory.dynamic.AbstractInternalDynamicInformationGenerator;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.WebHelper;

/**
 * Generator for saxing dynamic information for admin dashboard service
 */
public class FormAdminDashboardDynamicInformationGenerator extends AbstractInternalDynamicInformationGenerator
{
    /** The form DAO */
    protected FormDAO _formDAO;
    
    /** The right manager */
    protected RightManager _rightManager;
    
    /** The form admin dashboard helper */
    protected FormAdminDashboardHelper _formAdminDashboardHelper;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _formDAO = (FormDAO) serviceManager.lookup(FormDAO.ROLE);
        _rightManager = (RightManager) serviceManager.lookup(RightManager.ROLE);
        _formAdminDashboardHelper = (FormAdminDashboardHelper) serviceManager.lookup(FormAdminDashboardHelper.ROLE);
    }
    
    @Override
    protected void saxShortValue() throws SAXException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        String siteName = WebHelper.getSiteName(request);
        
        List<Form> formToAdmin = _formAdminDashboardHelper.getFormToAdmin(siteName, getCurrentUser());
        XMLUtils.createElement(contentHandler, SHORT_VALUE, String.valueOf(formToAdmin.size()));
    }

    @Override
    protected void saxLongValue() throws SAXException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        String siteName = WebHelper.getSiteName(request);
        
        List<Form> formToAdmin = _formAdminDashboardHelper.getFormToAdmin(siteName, getCurrentUser());
        int nb = formToAdmin.size();
        if (nb != 0)
        {
            List<String> params = new ArrayList<>();
            params.add(String.valueOf(nb));
            I18nizableText longValue = new I18nizableText("plugin.forms", nb > 1 ? "PLUGINS_FORMS_LINK_DIRECTORY_ADMIN_DASHBOARD_LONG_VALUE_MULTIPLE" : "PLUGINS_FORMS_LINK_DIRECTORY_ADMIN_DASHBOARD_LONG_VALUE", params);

            XMLUtils.startElement(contentHandler, LONG_VALUE);
            longValue.toSAX(contentHandler);
            XMLUtils.endElement(contentHandler, LONG_VALUE);
        }
        else
        {
            I18nizableText longValue = new I18nizableText("plugin.forms", "PLUGINS_FORMS_LINK_DIRECTORY_ADMIN_DASHBOARD_LONG_VALUE_NO_FORMS");
            XMLUtils.startElement(contentHandler, LONG_VALUE);
            longValue.toSAX(contentHandler);
            XMLUtils.endElement(contentHandler, LONG_VALUE);
        }
    }

    @Override
    protected void saxTooltips() throws SAXException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        String siteName = WebHelper.getSiteName(request);
        
        List<Form> formToAdmin = _formAdminDashboardHelper.getFormToAdmin(siteName, getCurrentUser());
        
        for (Form form : formToAdmin)
        {
            saxItem(form.getTitle(), form.getDescription(), null);
        }
    }
}
