/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.generators;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.plugins.forms.dao.FormEntryDAO.Sort;
import org.ametys.plugins.forms.data.Answer;
import org.ametys.plugins.forms.helper.FormAdminDashboardHelper;
import org.ametys.plugins.forms.helper.FormMailHelper;
import org.ametys.plugins.forms.repository.Form;
import org.ametys.plugins.forms.repository.FormEntry;
import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.plugins.forms.workflow.FormEntryCurrentStepExpression;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.provider.RequestAttributeWorkspaceSelector;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.MetadataExpression;
import org.ametys.plugins.repository.query.expression.NotExpression;
import org.ametys.plugins.repository.query.expression.UserExpression;
import org.ametys.plugins.workflow.dao.WorkflowStepDAO;
import org.ametys.web.WebHelper;

import com.opensymphony.workflow.loader.StepDescriptor;
import com.opensymphony.workflow.loader.WorkflowDescriptor;

/**
 * This class generates all the forms process information for current user
 */
public class FormAdminDashboardGenerator extends FormDashboardGenerator
{
    /** The default number of entries by page */
    private static int __DEFAULT_NB_ENTRIES_BY_PAGE = 10;
    
    /** The default sort order */
    private static String __DEFAULT_SORT_ORDER = "ascending";
    
    /** The user manager */
    protected UserManager _userManager;
    
    /** The form mail helper */
    protected FormMailHelper _formMailHelper;
    
    /** The form admin dashboard helper */
    protected FormAdminDashboardHelper _formAdminDashboardHelper;
    
    /** The workflow step DAO */
    protected WorkflowStepDAO _stepDAO;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _userManager = (UserManager) smanager.lookup(UserManager.ROLE);
        _formMailHelper = (FormMailHelper) smanager.lookup(FormMailHelper.ROLE);
        _formAdminDashboardHelper = (FormAdminDashboardHelper) smanager.lookup(FormAdminDashboardHelper.ROLE);
        _stepDAO = (WorkflowStepDAO) smanager.lookup(WorkflowStepDAO.ROLE);
    }
    
    @Override
    protected void _saxFormDashboardData(UserIdentity user) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "dashboard");
        
        Request request = ObjectModelHelper.getRequest(objectModel);
        // Retrieve the current workspace.
        String currentWsp = RequestAttributeWorkspaceSelector.getForcedWorkspace(request);
        try
        {
            // Force the workspace.
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, RepositoryConstants.DEFAULT_WORKSPACE);
        
            String siteName = WebHelper.getSiteName(request);
        
            String formId = request.getParameter("formId");
            // No need to sax the forms if a form is send in the request
            if (StringUtils.isBlank(formId))
            {
                List<Form> forms = _formAdminDashboardHelper.getFormToAdmin(siteName, user);
                // Sax the forms for user selection
                _saxForms(forms);
            }
            else
            {
                // Sax form entries
                _saxFormEntries(request, formId, user);
            }
        }
        finally
        {
            // Restore context
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, currentWsp);
        }

        XMLUtils.endElement(contentHandler, "dashboard");
    }
    
    /**
     * Sax the forms
     * @param forms the list of form
     * @throws SAXException if a saxing error occurred
     */
    protected void _saxForms(List<Form> forms) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "forms");
        for (Form form : forms)
        {
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("id", form.getId());
            XMLUtils.startElement(contentHandler, "form", attrs);
            
            XMLUtils.createElement(contentHandler, "title", form.getTitle());
            
            String description = form.getDescription();
            if (description != null)
            {
                XMLUtils.createElement(contentHandler, "description", description);
            }
            
            XMLUtils.endElement(contentHandler, "form");
        }
        XMLUtils.endElement(contentHandler, "forms");
    }
    
    /**
     * Get available form entries for given page
     * @param formId the form id
     * @param page the page
     * @param nbEntriesByPage the number of entries by page
     * @param sortOrder the sort order for entries, can be ascending or descending
     * @param stateId the filter for state id
     * @param submitterId the filter for submitter id
     * @return the entries record
     */
    protected EntriesRecord _getAvailableFormEntries(String formId, int page, int nbEntriesByPage, String sortOrder, Integer stateId, String submitterId)
    {
        List<FormEntry> availableEntries = new ArrayList<>();

        Form form = _resolver.resolveById(formId);
        
        Expression filterExpr = _getFilterExpression(stateId, submitterId);
        List<FormEntry> entries = _formEntryDAO.getFormEntries(form, false, filterExpr, List.of(new Sort("ametys-id", sortOrder)));
        
        int activeEntriesSize = entries.size();
        int nbAnswer = 0;
        int position = 0;
        int lastEntryIndex = page * nbEntriesByPage;
        boolean hasNextPage = false;
        while (nbAnswer <= lastEntryIndex && position < activeEntriesSize)
        {
            FormEntry entry = entries.get(position);
            if (_formAdminDashboardHelper.hasAvailableActions(entry))
            {
                if (nbAnswer == lastEntryIndex)
                {
                    hasNextPage = true;
                }
                else
                {
                    availableEntries.add(entry);
                }
                nbAnswer++;
            }
            position++;
        }
        
        return new EntriesRecord(availableEntries, hasNextPage);
    }
    
    private Expression _getFilterExpression(Integer stateId, String submitterId)
    {
        List<Expression> filterExprs = new ArrayList<>();
        if (StringUtils.isNotBlank(submitterId))
        {
            filterExprs.add(new UserExpression(FormEntry.ATTRIBUTE_USER, Operator.EQ, submitterId));
        }
        
        if (stateId != null)
        {
            filterExprs.add(new FormEntryCurrentStepExpression(stateId));
        }
        
        // filter anonymized entries
        filterExprs.add(new NotExpression(new MetadataExpression(FormEntry.ATTRIBUTE_ANONYMIZATION_DATE)));
        
        return new AndExpression(filterExprs);
    }
    
    /**
     * Sax the form entries
     * @param request the request
     * @param formId the form id
     * @param user the current user
     * @throws SAXException if a saxing error occurred
     */
    protected void _saxFormEntries(Request request, String formId, UserIdentity user) throws SAXException
    {
        Form form = _resolver.resolveById(formId);
        String stateIdAsString = request.getParameter("workflowFilter");
        Integer stateId = StringUtils.isNotBlank(stateIdAsString) ? Integer.parseInt(stateIdAsString) : null;
        String submitter = request.getParameter("submitterFilter");
        
        int nbAnswerByPage = StringUtils.isNotBlank(request.getParameter("nbEntriesByPage")) ? Integer.parseInt(request.getParameter("nbEntriesByPage")) : __DEFAULT_NB_ENTRIES_BY_PAGE;
        int page = request.getParameter("page") != null ? Integer.parseInt(request.getParameter("page")) : 1;
        String sortOrder = StringUtils.isNotBlank(request.getParameter("entriesSortOrder")) ? request.getParameter("entriesSortOrder") : __DEFAULT_SORT_ORDER;
        EntriesRecord availableFormEntries = _getAvailableFormEntries(formId, page, nbAnswerByPage, sortOrder, stateId, submitter);
        List<FormEntry> entries = availableFormEntries.entries();
        
        // Recalculate the page
        int size = entries.size();
        int mod = size % nbAnswerByPage;
        page = mod == 0 && size != 0 ? (size / nbAnswerByPage) : (size / nbAnswerByPage) + 1;
        List<FormEntry> entriesToSax = entries.subList((page - 1) * nbAnswerByPage, Math.min(page * nbAnswerByPage, size));
        
        // Sax entries
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("formId", formId);
        attrs.addCDATAAttribute("formTitle", form.getTitle());
        attrs.addCDATAAttribute("page", String.valueOf(page));
        attrs.addCDATAAttribute("hasNextPage", String.valueOf(availableFormEntries.hasNextEntries()));
        XMLUtils.startElement(contentHandler, "answers", attrs);
        
        // Sax filters for displaying entries
        _saxFilters(form, stateId, submitter);
        
        try
        {
            for (FormEntry entry : entriesToSax)
            {
                _saxAnswer(_formEntry2Answer(entry), user);
            }
        }
        catch (Exception e)
        {
            getLogger().error("An error occurred saxing answers", e);
        }
        XMLUtils.endElement(contentHandler, "answers");
    }
    
    /**
     * Sax the filters for entry search
     * @param form the form
     * @param stateId the state id. Can be null.
     * @param submitterId the submitter id. Can be null.
     * @throws SAXException if a saxing error occurred
     */
    protected void _saxFilters(Form form, Integer stateId, String submitterId) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "filters");
        _saxStateFilter(form, stateId);
        _saxSubmitterFilter(submitterId);
        XMLUtils.endElement(contentHandler, "filters");
        
    }

    private void _saxStateFilter(Form form, Integer stateId) throws SAXException
    {
        AttributesImpl workflowAttrs = new AttributesImpl();
        if (stateId != null)
        {
            workflowAttrs.addCDATAAttribute("value", String.valueOf(stateId));
        }
        
        XMLUtils.startElement(contentHandler, "workflow-states", workflowAttrs);
        
        String workflowName = form.getWorkflowName();
        WorkflowDescriptor workflowDescriptor = _workflowHelper.getWorkflowDescriptor(workflowName);
        List<StepDescriptor> steps = workflowDescriptor.getSteps();
        for (StepDescriptor step : steps)
        {
            AttributesImpl stepAttrs = new AttributesImpl();
            int stepId = step.getId();
            stepAttrs.addCDATAAttribute("value", String.valueOf(stepId));
            stepAttrs.addCDATAAttribute("label", _stepDAO.getStepLabelAsString(workflowDescriptor, stepId, false));
            XMLUtils.createElement(contentHandler, "state", stepAttrs);
        }
        
        XMLUtils.endElement(contentHandler, "workflow-states");
    }

    private void _saxSubmitterFilter(String submitterId) throws SAXException
    {
        AttributesImpl attrs = new AttributesImpl();
        if (StringUtils.isNotBlank(submitterId))
        {
            attrs.addCDATAAttribute("value", submitterId);
        }
        
        XMLUtils.createElement(contentHandler, "submitters", attrs);
    }
    
    @Override
    protected void _saxWorkflowInformations(Answer answer, UserIdentity user) throws Exception
    {
        super._saxWorkflowInformations(answer, user);
        
        FormEntry formEntry = _resolver.resolveById(answer.getId());
        _saxMailUsersInformations(formEntry, user);
    }
    
    private void _saxMailUsersInformations(FormEntry formEntry, UserIdentity user) throws SAXException
    {
        User currentUser = _userManager.getUser(user);
        
        XMLUtils.createElement(contentHandler, "sender-mail", currentUser.getEmail());
        XMLUtils.createElement(contentHandler, "sender-fullname", currentUser.getFullName() + " <" + currentUser.getEmail() + ">");
        
        XMLUtils.startElement(contentHandler, "receivers");
        User entryUser = _userManager.getUser(formEntry.getUser());
        if (entryUser != null)
        {
            XMLUtils.startElement(contentHandler, "receiver");
            XMLUtils.createElement(contentHandler, "receiver-mail", entryUser.getEmail());
            XMLUtils.createElement(contentHandler, "receiver-fullname", entryUser.getFullName() + " <" + entryUser.getEmail() + ">");
            XMLUtils.endElement(contentHandler, "receiver");
        }
        
        for (FormQuestion question : _formMailHelper.getQuestionWithMail(formEntry.getForm().getId()))
        {
            Optional<String> receiver = _formMailHelper.getReceiver(formEntry, Optional.of(question.getNameForForm()));
            if (receiver.isPresent())
            {
                XMLUtils.startElement(contentHandler, "receiver");
                XMLUtils.createElement(contentHandler, "receiver-mail", receiver.get());
                XMLUtils.createElement(contentHandler, "receiver-fullname", question.getTitle() + " <" + receiver.get() + ">");
                XMLUtils.endElement(contentHandler, "receiver");
            }
        }
        XMLUtils.endElement(contentHandler, "receivers");
    }
    
    private record EntriesRecord(List<FormEntry> entries, boolean hasNextEntries) { /** empty */ }
}
