/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.helper;

import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceResolver;

import org.ametys.cms.data.File;
import org.ametys.core.ui.Callable;
import org.ametys.core.ui.mail.StandardMailBodyHelper;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.core.util.I18nUtils;
import org.ametys.core.util.mail.SendMailHelper;
import org.ametys.core.util.mail.SendMailHelper.MailBuilder;
import org.ametys.core.util.mail.SendMailHelper.NamedStream;
import org.ametys.plugins.forms.dao.FormDAO;
import org.ametys.plugins.forms.dao.FormEntryDAO;
import org.ametys.plugins.forms.dao.FormQuestionDAO;
import org.ametys.plugins.forms.dao.FormQuestionDAO.FormEntryValues;
import org.ametys.plugins.forms.question.FormQuestionType;
import org.ametys.plugins.forms.question.sources.AbstractSourceType;
import org.ametys.plugins.forms.question.sources.ManualWithEmailSourceType;
import org.ametys.plugins.forms.question.sources.UsersSourceType;
import org.ametys.plugins.forms.question.types.impl.ChoicesListQuestionType;
import org.ametys.plugins.forms.question.types.impl.FileQuestionType;
import org.ametys.plugins.forms.question.types.impl.SimpleTextQuestionType;
import org.ametys.plugins.forms.repository.Form;
import org.ametys.plugins.forms.repository.FormEntry;
import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.plugins.forms.rights.FormsDirectoryRightAssignmentContext;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.site.Site;

import jakarta.mail.MessagingException;

/**
 * The helper for form mail dialog
 */
public class FormMailHelper extends AbstractLogEnabled implements Serviceable, Component, Contextualizable
{

    /** Avalon ROLE. */
    public static final String ROLE = FormMailHelper.class.getName();
   
    /** The param key for read restriction enable */
    public static final String READ_RESTRICTION_ENABLE = "read-restriction-enable";
    
    /** The value for entry user in the receiver combobox */
    public static final String RECEIVER_COMBOBOX_ENTRY_USER_VALUE = "entry-user";
    
    /** The empty value in the receiver combobox */
    public static final String RECEIVER_COMBOBOX_INPUT_ONLY = "input-only";

    /** The request key to ignore right */
    public static final String IGNORE_RIGHT_KEY = "ignore-right";

    /** Pattern for adding entry in acknowledgement of receipt if present in body */
    protected static final String _FORM_ENTRY_PATTERN = "{form}";
    
    /** Ametys object resolver. */
    protected AmetysObjectResolver _resolver;
    
    /** I18n Utils */
    protected I18nUtils _i18nUtils;
    
    /** The user manager */
    protected UserManager _userManager;
    
    /** The source resolver. */
    protected SourceResolver _sourceResolver;
    
    /** The context */
    protected Context _context;

    /** The analyse of files for virus helper */
    protected FormDAO _formDAO;
    
    /** The form question DAO */
    protected FormQuestionDAO _formQuestionDAO;
    
    /** The form entry DAO */
    protected FormEntryDAO _formEntryDAO;
    
    /** The form admin mail helper */
    protected FormAdminMailsHelper _formAdminMailsHelper;
    
    /**
     * The type of limitation for the mail
     */
    public enum LimitationMailType
    {
        /** The mail for queue */
        QUEUE,
        /** The mail for the limit */
        LIMIT;
    }
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _sourceResolver = (SourceResolver) manager.lookup(SourceResolver.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _i18nUtils = (I18nUtils) manager.lookup(I18nUtils.ROLE);
        _userManager = (UserManager) manager.lookup(UserManager.ROLE);
        _formDAO = (FormDAO) manager.lookup(FormDAO.ROLE);
        _formQuestionDAO = (FormQuestionDAO) manager.lookup(FormQuestionDAO.ROLE);
        _formEntryDAO = (FormEntryDAO) manager.lookup(FormEntryDAO.ROLE);
        _formAdminMailsHelper = (FormAdminMailsHelper) manager.lookup(FormAdminMailsHelper.ROLE);
    }
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    /**
     * Get the fields with email regex constraint so they can be used as receivers address for form admin mail
     * @param formId Id of the form
     * @return a map of the form question, key is the question id,value is the question title
     */
    @Callable (rights = FormDAO.HANDLE_FORMS_RIGHT_ID, rightContext = FormsDirectoryRightAssignmentContext.ID, paramIndex = 0)
    public Map<String, Object> getAvailableAdminReceiverFields(String formId)
    {
        List<Object> receiverfields = new ArrayList<>();
        
        for (FormQuestion question : getQuestionWithMail(formId))
        {
            Map<String, String> properties = new HashMap<>();
            properties.put("id", question.getNameForForm());
            properties.put("title", question.getTitle());
            receiverfields.add(properties);
        }
        
        return Map.of("data", receiverfields);
    }
    
    /**
     * Get the fields with email regex constraint so they can be used as receivers address for form receipt mail
     * @param formId Id of the form
     * @return a map of the form question, key is the question id,value is the question title
     */
    @SuppressWarnings("unchecked")
    @Callable (rights = FormDAO.HANDLE_FORMS_RIGHT_ID, rightContext = FormsDirectoryRightAssignmentContext.ID, paramIndex = 0)
    public Map<String, Object> getAvailableReceiverFields(String formId)
    {
        Map<String, Object> results = getAvailableAdminReceiverFields(formId);
        List<Object> receiverfields = (List<Object>) results.get("data");
        
        Map<String, String> properties = new HashMap<>();
        properties.put("id", RECEIVER_COMBOBOX_ENTRY_USER_VALUE);
        properties.put("title", _i18nUtils.translate(new I18nizableText("plugin.forms", "PLUGINS_FORMS_ACKNOWLEDGEMENT_RECEIPT_ENTRY_USER")));
        receiverfields.add(properties);
        
        return Map.of("data", receiverfields);
    }
    
    /**
     * Get the list of questions which return a mail
     * @param formId the form id
     * @return the list of questions
     */
    public List<FormQuestion> getQuestionWithMail(String formId)
    {
        List<FormQuestion> questions = new ArrayList<>();
        Form form = _resolver.resolveById(formId);
        
        for (FormQuestion question : form.getQuestions())
        {
            FormQuestionType type = question.getType();
            if (type instanceof SimpleTextQuestionType)
            {
                if (question.hasValue(SimpleTextQuestionType.ATTRIBUTE_REGEXP) && question.getValue(SimpleTextQuestionType.ATTRIBUTE_REGEXP).equals(SimpleTextQuestionType.EMAIL_REGEX_VALUE))
                {
                    questions.add(question);
                }
            }
            else if (type instanceof ChoicesListQuestionType cLType && (cLType.getSourceType(question) instanceof UsersSourceType || cLType.getSourceType(question) instanceof ManualWithEmailSourceType))
            {
                questions.add(question);
            }
        }
        
        return questions;
    }
    
    /**
     * Get all the email addresses from manual entry and/or a form field
     * @param form the form
     * @param entry the last entry
     * @return an array of all the admin emails
     */
    public String[] getAdminEmails(Form form, FormEntry entry)
    {
        Optional<String[]> adminEmails = form.getAdminEmails();
        Optional<String> otherAdminEmails = form.getOtherAdminEmails();
        String[] emailsAsArray = adminEmails.isPresent() ? adminEmails.get() : ArrayUtils.EMPTY_STRING_ARRAY;
        Optional<String> otherEmailSource = getReceiver(entry, otherAdminEmails);
        String[] mergedEmails = emailsAsArray;
        if (otherEmailSource.isPresent())
        {
            String otherEmails = otherEmailSource.get();
            String[] emails = otherEmails.split("[ ,;\r]");
            List<String> validEmails = _formAdminMailsHelper.getValidAdminEmails(emails);
            if (!validEmails.isEmpty())
            {
                mergedEmails = ArrayUtils.addAll(emailsAsArray, validEmails.toArray(new String[validEmails.size()]));
            }
            else
            {
                getLogger().error("Mails addresses " + otherEmails + " did not match regex");
            }
        }
        return mergedEmails;
    }
    
    /**
     * Get the receiver from the entry
     * @param entry the entry
     * @param receiverPath the path to get receiver
     * @return the receiver from the entry
     */
    public Optional<String> getReceiver(FormEntry entry, Optional<String> receiverPath)
    {
        if (receiverPath.isEmpty())
        {
            return Optional.empty();
        }
        
        if (receiverPath.get().equals(RECEIVER_COMBOBOX_ENTRY_USER_VALUE))
        {
            UserIdentity userIdentity = entry.getUser();
            User user = _userManager.getUser(userIdentity);
            return Optional.ofNullable(user != null ? user.getEmail() : null);
        }
        else
        {
            FormQuestion question = entry.getForm().getQuestion(receiverPath.get());
            FormQuestionType type = question.getType();
            if (type instanceof SimpleTextQuestionType)
            {
                return Optional.ofNullable(entry.getValue(receiverPath.get()));
            }
            else if (type instanceof ChoicesListQuestionType cLType)
            {
                if (cLType.getSourceType(question) instanceof UsersSourceType && !entry.isMultiple(receiverPath.get()))
                {
                    UserIdentity userIdentity = entry.getValue(receiverPath.get());
                    if (userIdentity != null)
                    {
                        User user = _userManager.getUser(userIdentity);
                        return user != null ? Optional.ofNullable(user.getEmail()) : Optional.empty();
                    }
                }
                else if (cLType.getSourceType(question) instanceof ManualWithEmailSourceType manualWithEmail)
                {
                    String value = entry.getValue(receiverPath.get());
                    Map<String, Object> enumParam = new HashMap<>();
                    enumParam.put(AbstractSourceType.QUESTION_PARAM_KEY, question);
                    try
                    {
                        return Optional.ofNullable(manualWithEmail.getEntryEmail(value, enumParam));
                    }
                    catch (Exception e)
                    {
                        getLogger().error("Could not get email from question '" + question.getNameForForm() + "' with value '" + value + "'");
                    }
                }
            }
            
            return Optional.empty();
        }
    }
    
    /**
     * Send the notification emails.
     * @param form the form.
     * @param entry the user input.
     * @param adminEmails list of email address where to send the notification
     */
    public void sendEmailsForAdmin(Form form, FormEntry entry, String[] adminEmails)
    {
        try
        {
            String lang = StringUtils.defaultIfBlank(form.getAdminEmailLanguage().orElse(null), _formDAO.getFormLocale(form));
            String sender = Config.getInstance().getValue("smtp.mail.from");
            UserIdentity userIdentity = entry.getUser();
            User user = userIdentity != null ? _userManager.getUser(userIdentity) : null;
            
            // Get subject
            Optional<String> adminEmailSubject = form.getAdminEmailSubject();
            I18nizableText subject = adminEmailSubject.isPresent()
                    ? _replaceVariablesAndBreaks(form, user, adminEmailSubject.get(), lang)
                    : null;
            
            // Get body with details
            Optional<String> adminEmailBody = form.getAdminEmailBody();
            I18nizableText message = adminEmailBody.isPresent()
                    ? _replaceVariablesAndBreaks(form, user, adminEmailBody.get(), lang)
                    : null;

            String entryDetails = getMail("entry.html", entry, Map.of(), false);
            
            String prettyHtmlBody = StandardMailBodyHelper.newHTMLBody()
                    .withLanguage(lang)
                    .withTitle(subject)
                    .withMessage(message)
                    .withDetails(new I18nizableText("plugin.forms", "PLUGINS_FORMS_MAIL_RESULTS_DETAILS_TITLE"), entryDetails, false)
                    .build();

            // Get mail as text
            String params = "?type=results&form-name=" + form.getTitle() + "&locale=" + lang;
            String text = getMail("results.txt" + params, entry, Map.of(), false);
           
            // Send mails
            for (String email : adminEmails)
            {
                if (StringUtils.isNotEmpty(email))
                {
                    _sendMail(form, entry, _i18nUtils.translate(subject, lang), prettyHtmlBody, text, sender, email, false, false);
                }
            }
        }
        catch (IOException e)
        {
            getLogger().error("Error creating the notification message.", e);
        }
    }

    private I18nizableText _replaceVariablesAndBreaks(Form form, User user, String mailText, String language)
    {
        String text = mailText;
        text = StringUtils.replace(text, "{site}", form.getSite().getTitle());
        text = StringUtils.replace(text, "{user}",  user != null ? user.getFullName() : _i18nUtils.translate(new I18nizableText("plugin.forms", "PLUGINS_FORMS_ADMIN_EMAILS_USER_ANONYMOUS"), language));
        text = StringUtils.replace(text, "{title}",  form.getTitle());
        text = text.replaceAll("\r?\n", "<br/>");
        return new I18nizableText(text);
    }
    
    /**
     * Send limitation mail when the limit is reached
     * @param entry the form entry
     * @param adminEmails list of email address where to send the notification
     * @param limitationType the type of limitation
     */
    public void sendLimitationReachedMailForAdmin(FormEntry entry, String[] adminEmails, LimitationMailType limitationType)
    {
        try
        {
            Form form = entry.getForm();
            
            String lang = StringUtils.defaultIfBlank(form.getAdminEmailLanguage().orElse(null), _formDAO.getFormLocale(form));
            String sender = Config.getInstance().getValue("smtp.mail.from");
            
            // Get subject
            I18nizableText subject = limitationType == LimitationMailType.LIMIT
                    ? new I18nizableText("plugin.forms", "PLUGINS_FORMS_MAIL_LIMIT_SUBJECT", List.of(form.getTitle(), form.getSite().getTitle()))
                    : new I18nizableText("plugin.forms", "PLUGINS_FORMS_MAIL_QUEUE_SUBJECT", List.of(form.getTitle(), form.getSite().getTitle()));
            
            // Get body
            I18nizableText message = limitationType == LimitationMailType.LIMIT
                    ? new I18nizableText("plugin.forms", "PLUGINS_FORMS_MAIL_LIMIT_TEXT", List.of(form.getTitle()))
                    : new I18nizableText("plugin.forms", "PLUGINS_FORMS_MAIL_QUEUE_TEXT", List.of(form.getTitle()));
            
            String prettyHtmlBody = StandardMailBodyHelper.newHTMLBody()
                    .withLanguage(lang)
                    .withTitle(subject)
                    .withMessage(message)
                    .build();
            
            // Get mail as text
            I18nizableText textI18n = limitationType == LimitationMailType.LIMIT
                    ? new I18nizableText("plugin.forms", "PLUGINS_FORMS_MAIL_LIMIT_TEXT_NO_HTML", List.of(form.getTitle()))
                    : new I18nizableText("plugin.forms", "PLUGINS_FORMS_MAIL_QUEUE_TEXT_NO_HTML", List.of(form.getTitle()));
            String text = _i18nUtils.translate(textI18n, lang);
            
            for (String email : adminEmails)
            {
                if (StringUtils.isNotEmpty(email))
                {
                    _sendMail(form, entry, _i18nUtils.translate(subject, lang), prettyHtmlBody, text, sender, email, false, false);
                }
            }
        }
        catch (IOException e)
        {
            getLogger().error("Error creating the limit message.", e);
        }
    }
    
    /**
     * Send the receipt email.
     * @param form the form.
     * @param entry the current entry
     */
    public void sendReceiptEmail(Form form, FormEntry entry)
    {
        if (form.getReceiptSender().isPresent())
        {
            Optional<String> receiver = getReceiver(entry, form.getReceiptReceiver());
            if (receiver.isPresent())
            {
                String lang = _formDAO.getFormLocale(form);
                
                String sender = form.getReceiptSender().get();
                String subject = form.getReceiptSubject().get();
                String bodyTxt = form.getReceiptBody().get();
                String bodyHTML = bodyTxt.replaceAll("\r?\n", "<br/>");
                
                try
                {
                    String prettyHtmlBody = StandardMailBodyHelper.newHTMLBody()
                            .withLanguage(lang)
                            .withTitle(subject)
                            .withMessage(bodyHTML)
                            .build();

                    // Always check reading restriction for the receipt email
                    _sendMail(form, entry, subject, prettyHtmlBody, bodyTxt, sender, receiver.get(), true, true);
                }
                catch (IOException e)
                {
                    getLogger().error("An error occurred sending receipt mail to '{}'", receiver.get(), e);
                }
            }
        }
    }
    
    /**
     * Send mail when the form entry if out of the queue
     * @param form the form
     * @param entry the form entry
     */
    public void sendOutOfQueueMail(Form form, FormEntry entry)
    {
        Optional<String> receiver = getReceiver(entry, form.getQueueMailReceiver());
        if (receiver.isPresent())
        {
            String lang = _formDAO.getFormLocale(form);
            
            String sender = form.getQueueMailSender().get();
            String subject = form.getQueueMailSubject().get();
            String bodyTxt = form.getQueueMailBody().get();
            String bodyHTML = bodyTxt.replaceAll("\r?\n", "<br/>");
            
            try
            {
                String prettyHtmlBody = StandardMailBodyHelper.newHTMLBody()
                        .withLanguage(lang)
                        .withTitle(subject)
                        .withMessage(bodyHTML)
                        .build();

                // Always check reading restriction for the receipt email
                _sendMail(form, entry, subject, prettyHtmlBody, bodyTxt, sender, receiver.get(), true, true);
            }
            catch (IOException e)
            {
                getLogger().error("An error occurred sending mail to get out of the queue to '{}'", receiver.get(), e);
            }
        }
    }
    
    /**
     * Send invitation mails to the users
     * @param form the form
     * @param users the users to receive the mail
     * @param message the invitation message
     * @param language The language to use in the mail. Can be null, default to form's language
     */
    public void sendInvitationMails(Form form, List<User> users, String message, String language)
    {
        String formLang = _formDAO.getFormLocale(form);
        String lang = StringUtils.defaultIfBlank(language, formLang);
        String sender = Config.getInstance().getValue("smtp.mail.from");
        
        // Get subject
        I18nizableText subject = new I18nizableText("plugin.forms", "PLUGINS_FORMS_SEND_INVITATIONS_BOX_SUBJECT");
        
        // Get body
        Site site = form.getSite();
        String formURI = _getFormURI(form, formLang);
        
        String replacedMessage = StringUtils.replace(message, "{site}", site.getTitle());
        String textMessage = StringUtils.replace(replacedMessage, "{link}", formURI);
        String htmlMessage = StringUtils.replace(replacedMessage, "{link}", "<a href='" + formURI + "'>" + formURI + "</a>");
        
        for (User user : users)
        {
            try
            {
                String finalTextMessage = StringUtils.replace(textMessage, "{name}", user.getFullName());
                String finalHtmlMessage = StringUtils.replace(htmlMessage, "{name}", user.getFullName());
                String prettyHtmlBody = StandardMailBodyHelper.newHTMLBody()
                        .withLanguage(lang)
                        .withTitle(subject)
                        .withMessage(finalHtmlMessage)
                        .withLink(formURI, new I18nizableText("plugin.forms", "PLUGINS_FORMS_SEND_INVITATIONS_LINK_LABEL"))
                        .build();
                
                _sendMail(form, null, _i18nUtils.translate(subject, lang), prettyHtmlBody, finalTextMessage, sender, user.getEmail(), false, true);
            }
            catch (IOException e)
            {
                getLogger().error("Unable to send invitation mail to user {}", user.getEmail(), e);
            }
        }
    }
    
    /**
     * Get the form page URI
     * @param form the form
     * @param language the language
     * @return the form page URI
     */
    protected String _getFormURI (Form form, String language)
    {
        Site site = form.getSite();
        Optional<Page> page = _formDAO.getFormPage(form.getId(), site.getName())
            .stream()
            .filter(Page.class::isInstance)
            .map(Page.class::cast)
            .filter(p -> p.getSitemapName().equals(language))
            .findAny();
        
        return page.map(p -> site.getUrl() + "/" + p.getSitemap().getName() + "/" + p.getPathInSitemap() + ".html")
                   .orElse(null);
    }
    
    /**
     * Get a mail pipeline's content.
     * @param resource the mail resource pipeline
     * @param entry the user input.
     * @param additionalParameters the additional parameters
     * @param withReadingRestriction <code>true</code> to enable reading restriction for form data in the mail
     * @return the mail content.
     * @throws IOException if an error occurs.
     */
    public String getMail(String resource, FormEntry entry, Map<String, Object> additionalParameters, boolean withReadingRestriction) throws IOException
    {
        Source src = null;
        Request request = ContextHelper.getRequest(_context);
        Form form = entry.getForm();
        
        try
        {
            request.setAttribute(IGNORE_RIGHT_KEY, true);
            
            String uri = "cocoon:/mail/entry/" + resource;
            Map<String, Object> parameters = new HashMap<>();
            parameters.put("formId", form.getId());
            parameters.put("entryId", entry.getId());
            parameters.put(READ_RESTRICTION_ENABLE, withReadingRestriction);

            parameters.putAll(additionalParameters);
            
            src = _sourceResolver.resolveURI(uri, null, parameters);
            Reader reader = new InputStreamReader(src.getInputStream(), "UTF-8");
            return IOUtils.toString(reader);
        }
        finally
        {
            request.setAttribute(IGNORE_RIGHT_KEY, false);
            _sourceResolver.release(src);
        }
    }
    
    /**
     * Get the files of a user input.
     * @param form The current form
     * @param entry The current entry
     * @param onlyWritableQuestion <code>true</code> to have only writable question
     * @param onlyReadableQuestion <code>true</code> to have only readable question
     * @return the files submitted by the user.
     */
    public Collection<NamedStream> getEntryFiles(Form form, FormEntry entry, boolean onlyWritableQuestion, boolean onlyReadableQuestion)
    {
        Optional<Long> currentStepId = entry.getForm().hasWorkflow() ? Optional.of(_formEntryDAO.getCurrentStepId(entry)) : Optional.empty();
        return _formQuestionDAO.getRuleFilteredQuestions(form, new FormEntryValues(null, entry), currentStepId, onlyWritableQuestion, onlyReadableQuestion).stream()
            .filter(q -> q.getType() instanceof FileQuestionType)
            .map(q -> entry.<File>getValue(q.getNameForForm()))
            .filter(Objects::nonNull)
            .map(f -> new NamedStream(f.getInputStream(), f.getName(), f.getMimeType()))
            .collect(Collectors.toList());
    }
    
    private void _sendMail(Form form, FormEntry entry, String subject, String html, String text, String sender, String recipient, boolean addFormInformation, boolean withReadingRestriction)
    {
        try
        {
            String htmlBody = html;
            String textBody = text;
        
            Collection<NamedStream> records = entry != null ? getEntryFiles(form, entry, false, false) : new ArrayList<>();
            try
            {
                if (addFormInformation)
                {
                    if (textBody.contains(_FORM_ENTRY_PATTERN))
                    {
                        String entry2text = getMail("entry.txt", entry, Map.of(), withReadingRestriction);
                        textBody = StringUtils.replace(textBody, _FORM_ENTRY_PATTERN, entry2text);
                    }
                    
                    if (htmlBody.contains(_FORM_ENTRY_PATTERN))
                    {
                        String entry2html = getMail("entry.html", entry, Map.of(), withReadingRestriction);
                        htmlBody = StringUtils.replace(htmlBody, _FORM_ENTRY_PATTERN, entry2html);
                    }
                }
                
                MailBuilder mailBuilder = SendMailHelper.newMail()
                    .withAsync(true)
                    .withSubject(subject)
                    .withHTMLBody(htmlBody)
                    .withTextBody(textBody)
                    .withSender(sender)
                    .withRecipient(recipient);
                
                if (!records.isEmpty())
                {
                    mailBuilder = mailBuilder.withAttachmentsAsStream(records);
                }
                mailBuilder.sendMail();
            }
            finally
            {
                for (NamedStream record : records)
                {
                    IOUtils.close(record.inputStream());
                }
            }
        }
        catch (MessagingException | IOException e)
        {
            getLogger().error("Error sending the mail to " + recipient, e);
        }
    }
}
