/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.question.types.impl;

import java.time.ZonedDateTime;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.data.Binary;
import org.ametys.cms.data.holder.DataHolderRelativeDisableCondition;
import org.ametys.cms.data.holder.DataHolderRelativeDisableConditions;
import org.ametys.plugins.forms.question.types.AbstractFormQuestionType;
import org.ametys.plugins.forms.question.validators.FileFormValidator;
import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.SimpleViewItemGroup;
import org.ametys.runtime.model.StaticEnumerator;
import org.ametys.runtime.model.ViewElement;
import org.ametys.runtime.model.disableconditions.DisableCondition;
import org.ametys.runtime.model.disableconditions.DisableCondition.OPERATOR;
import org.ametys.runtime.model.disableconditions.DisableConditions;
import org.ametys.runtime.model.type.ModelItemTypeConstants;
import org.ametys.runtime.parameter.DefaultValidator;
import org.ametys.runtime.servlet.RuntimeServlet;

/**
 * Class for creating file questions
 */
public class FileQuestionType extends AbstractFormQuestionType
{
    /** Constant for max size attribute. */
    public static final String ATTRIBUTE_MAX_SIZE = "max-size";
    
    /** Constant for allowed extensions attribute. */
    public static final String ATTRIBUTE_EXTENSIONS = "extensions";
    
    /** Constant for other allowed extensions attribute. */
    public static final String ATTRIBUTE_EXTENSIONS_OTHER = "extensions-other";
    
    /** Constant for default title */
    public static final String DEFAULT_TITLE = "PLUGIN_FORMS_QUESTION_DEFAULT_TITLE_FILE";
    
    /** Constant for all extensions */
    public static final String ALL_EXTENSIONS_VALUE = "_ametys_all_extension";
    
    /** Constant for image extensions */
    public static final String IMAGE_EXTENSIONS_VALUE = "_ametys_image_extension";
    
    /** Constant for video extensions */
    public static final String VIDEO_EXTENSIONS_VALUE = "_ametys_video_extension";
    
    /** Constant for audio extensions */
    public static final String AUDIO_EXTENSIONS_VALUE = "_ametys_audio_extension";
    
    /** Constant for document extensions */
    public static final String DOCUMENT_EXTENSIONS_VALUE = "_ametys_document_extension";
    
    /** Constant for other extensions */
    public static final String OTHER_EXTENSIONS_VALUE = "_ametys_other_extension";
    
    @Override
    protected List<ModelItem> _getModelItems()
    {
        List<ModelItem> modelItems = super._getModelItems();
        
        ElementDefinition<Long> maxSize = _formElementDefinitionHelper.getElementDefinition(ATTRIBUTE_MAX_SIZE, ModelItemTypeConstants.LONG_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_FILE_MAX_SIZE", "PLUGINS_FORMS_QUESTIONS_DIALOG_FILE_MAX_SIZE_DESC", null);
        
        // Upload initialization
        long maxSizeValue = _getMaxSizeValue();
        maxSize.setWidget("edition.longfield");
        Map<String, I18nizableText> widgetParameters  = new HashMap<>();
        widgetParameters.put("maxValue", new I18nizableText(String.valueOf(maxSizeValue)));
        widgetParameters.put("minValue", new I18nizableText("1"));
        widgetParameters.put("emptyText", new I18nizableText(String.valueOf(maxSizeValue)));
        maxSize.setWidgetParameters(widgetParameters);
        modelItems.add(maxSize);
        
        ElementDefinition<String> extensions = _formElementDefinitionHelper.getElementDefinition(ATTRIBUTE_EXTENSIONS, ModelItemTypeConstants.STRING_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_FILE_EXTENSIONS", "PLUGINS_FORMS_QUESTIONS_DIALOG_FILE_EXTENSIONS_DESC", null);

        StaticEnumerator<String> extensionEnum = new StaticEnumerator<>();
        extensionEnum.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_FILE_EXTENSIONS_ALL"), ALL_EXTENSIONS_VALUE);
        extensionEnum.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_FILE_EXTENSIONS_DOCUMENTS"), DOCUMENT_EXTENSIONS_VALUE);
        extensionEnum.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_FILE_EXTENSIONS_IMAGES"), IMAGE_EXTENSIONS_VALUE);
        extensionEnum.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_FILE_EXTENSIONS_VIDEO"), VIDEO_EXTENSIONS_VALUE);
        extensionEnum.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_FILE_EXTENSIONS_AUDIO"), AUDIO_EXTENSIONS_VALUE);
        extensionEnum.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_FILE_EXTENSIONS_OTHER"), OTHER_EXTENSIONS_VALUE);
        extensions.setEnumerator(extensionEnum);
        extensions.setDefaultValue(ALL_EXTENSIONS_VALUE);
        
        Map<String, I18nizableText> extWidgetParameters  = new HashMap<>();
        extWidgetParameters.put("naturalOrder", new I18nizableText("true"));
        extensions.setWidgetParameters(extWidgetParameters);
        
        modelItems.add(extensions);
        
        ElementDefinition otherExtensions = _formElementDefinitionHelper.getElementDefinition(ATTRIBUTE_EXTENSIONS_OTHER, ModelItemTypeConstants.STRING_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_FILE_OTHER_EXTENSIONS", "PLUGINS_FORMS_QUESTIONS_DIALOG_FILE_OTHER_EXTENSIONS_DESC", new DefaultValidator("^\\.[a-z0-9]+(,\\s*\\.[a-z0-9]+)*$", false));
        DisableConditions disableConditions = new DataHolderRelativeDisableConditions();
        DisableCondition condition = new DataHolderRelativeDisableCondition(ATTRIBUTE_EXTENSIONS, OPERATOR.NEQ, "_ametys_other_extension", _disableConditionsHelper);
        disableConditions.getConditions().add(condition);
        otherExtensions.setDisableConditions(disableConditions);
        Map<String, I18nizableText> otherExtWidgetParameters  = new HashMap<>();
        otherExtWidgetParameters.put("ametysShowMultipleHint", new I18nizableText("true"));
        otherExtensions.setWidgetParameters(otherExtWidgetParameters);
        modelItems.add(otherExtensions);
        
        return modelItems;
    }
    
    @Override
    protected SimpleViewItemGroup _getMainTab()
    {
        SimpleViewItemGroup mainFieldset = super._getMainTab();
        
        ViewElement extensions = new ViewElement();
        extensions.setDefinition((ElementDefinition< ? >) getModel().getModelItem(ATTRIBUTE_EXTENSIONS));
        mainFieldset.addViewItem(extensions);
        
        ViewElement otherExtensions = new ViewElement();
        otherExtensions.setDefinition((ElementDefinition< ? >) getModel().getModelItem(ATTRIBUTE_EXTENSIONS_OTHER));
        mainFieldset.addViewItem(otherExtensions);
        
        return mainFieldset;
    }
    
    @Override
    protected SimpleViewItemGroup _getAdvancedTab()
    {
        SimpleViewItemGroup advancedFieldset =  super._getAdvancedTab();
        
        ViewElement maxSize = new ViewElement();
        maxSize.setDefinition((ElementDefinition< ? >) getModel().getModelItem(ATTRIBUTE_MAX_SIZE));
        advancedFieldset.addViewItem(maxSize);
        
        return advancedFieldset;
    }
    
    @Override
    public void validateQuestionValues(Map<String, Object> values, Map<String, I18nizableText> errors)
    {
        super.validateQuestionValues(values, errors);
        
        Number sizeAsNumber = (Number) values.getOrDefault(ATTRIBUTE_MAX_SIZE, null);
        Long maxSize = _getMaxSizeValue();
        Long size = sizeAsNumber != null ? sizeAsNumber.longValue() : maxSize;
        if (size > maxSize)
        {
            List<String> errorParams = Arrays.asList(String.valueOf(maxSize));
            
            errors.put(ATTRIBUTE_MAX_SIZE, new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_FILE_MAXSIZE_ERROR", errorParams));
        }
    }
    
    private long _getMaxSizeValue()
    {
        long maxSizeValue = RuntimeServlet.DEFAULT_MAX_UPLOAD_SIZE;
        Long maxUploadSizeParam = Config.getInstance().getValue("runtime.upload.max-size");
        if (maxUploadSizeParam != null)
        {
            maxSizeValue = maxUploadSizeParam.intValue();
        }
        
        return maxSizeValue / (1024 * 1024);
    }
    
    public String getStorageType(FormQuestion question)
    {
        return org.ametys.cms.data.type.ModelItemTypeConstants.FILE_ELEMENT_TYPE_ID;
    }
    
    @Override
    protected ModelItem _getEntryModelItem(FormQuestion question)
    {
        ModelItem item = super._getEntryModelItem(question);
        ((ElementDefinition) item).setValidator(new FileFormValidator(null, isMandatory(question), question.getValue(ATTRIBUTE_MAX_SIZE, false, _getMaxSizeValue()), _getAllowedExtensions(question)));
        return item;
    }
    
    @Override
    public void saxAdditionalInfos(ContentHandler contentHandler, FormQuestion question) throws SAXException
    {
        super.saxAdditionalInfos(contentHandler, question);
        
        XMLUtils.createElement(contentHandler, "allowed-extensions", _getAllowedExtensions(question));
        
        Long maxSize = question.getValue(ATTRIBUTE_MAX_SIZE, false, _getMaxSizeValue());
        XMLUtils.createElement(contentHandler, "max-size", String.valueOf(maxSize));
    }
    
    private String _getAllowedExtensions(FormQuestion question)
    {
        String value = question.getValue(ATTRIBUTE_EXTENSIONS, false, ALL_EXTENSIONS_VALUE);
        switch (value)
        {
            case DOCUMENT_EXTENSIONS_VALUE:
                return ".txt, .pdf, .xls, .xlsx, .doc, .docx, .ptt, .pttx, .odt, .ods, .odp, .odb, .odg, .odf";
            case IMAGE_EXTENSIONS_VALUE:
                return ".png, .gif, .jpg, .jpeg";
            case VIDEO_EXTENSIONS_VALUE:
                return ".mp4, .m4v, .mov, .avi, .flv, .wmv, .mpeg, .mkv";
            case AUDIO_EXTENSIONS_VALUE:
                return ".wav, .wave, .flac, .m4a, .wma, .wmv, .mp3, .ogg, .oga, .mogg, .aac, .aiff, .aif";
            case OTHER_EXTENSIONS_VALUE:
                return question.getValue(ATTRIBUTE_EXTENSIONS_OTHER, false, StringUtils.EMPTY);
            case ALL_EXTENSIONS_VALUE:
            default:
                return StringUtils.EMPTY;
        }
    }
    
    public I18nizableText getDefaultTitle()
    {
        return new I18nizableText("plugin.forms", DEFAULT_TITLE);
    }
    
    @Override
    public List<String> getFieldToDisableIfFormPublished(FormQuestion question)
    {
        List<String> fieldNames =  super.getFieldToDisableIfFormPublished(question);
        fieldNames.add(ATTRIBUTE_EXTENSIONS);
        fieldNames.add(ATTRIBUTE_EXTENSIONS_OTHER);
        return fieldNames;
    }
    
    public Map<String, Object> getAnonymizedData(FormQuestion question)
    {
        if (isMandatory(question))
        {
            Binary binary = new Binary();
            binary.setFilename("anonymized.txt");
            binary.setLastModificationDate(ZonedDateTime.now());
            
            return Map.of(question.getNameForForm(), binary);
        }
        else
        {
            return Collections.singletonMap(question.getNameForForm(), null);
        }
    }
}
