/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.question.validators;

import java.util.Collections;

import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.data.Binary;
import org.ametys.plugins.repository.data.holder.values.UntouchedValue;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.parameter.DefaultValidator;
import org.ametys.runtime.parameter.ValidationResult;
/**
 * Default validator for input files in Forms 
 */
public class FileFormValidator extends DefaultValidator
{
    /** The max size */
    protected Long _max;
    
    /** The allowed extensions */
    protected String _extensions;
    
    /**
     * A file form validator
     * @param regexp the regexp
     * @param mandatory true is mandatory
     * @param max the max size
     * @param extensions the allowed extensions
     */
    public FileFormValidator (String regexp, boolean mandatory, Long max, String extensions)
    {
        super(regexp, mandatory);
        _max = max;
        _extensions = extensions;
    }
    
    @Override
    protected ValidationResult validateSingleValue(Object value)
    {
        if (value instanceof UntouchedValue)
        {
            return ValidationResult.empty();
        }
        
        ValidationResult result = super.validateSingleValue(value);

        if (value != null)
        {
            Binary file = (Binary) value;

            // Validate file extensions.
            if (StringUtils.isNotBlank(_extensions))
            {
                String[] fileExtArray = _extensions.split(",");

                boolean extensionOk = false;
                for (int i = 0; i < fileExtArray.length && !extensionOk; i++)
                {
                    String ext = fileExtArray[i].trim().toLowerCase();
                    extensionOk = file.getName().toLowerCase().endsWith(ext);
                }

                if (!extensionOk)
                {
                    result.addError(new I18nizableText("plugin.forms", "PLUGINS_FORMS_FORMS_RENDER_ERROR_FILE_EXTENSION"));
                }
            }

            float maxLength = _max;
            if (maxLength < Float.MAX_VALUE)
            {
                //conversion from mo to bytes
                maxLength = maxLength * 1024 * 1024;
            }

            if (file.getLength() > maxLength)
            {
                result.addError(new I18nizableText("plugin.forms", "PLUGINS_FORMS_FORMS_RENDER_ERROR_FILE_TOOLARGE", Collections.singletonList(_max.toString() + "Mo")));
            }
        }

        return result;
    }
    
    @Override
    protected ValidationResult validateArrayValues(Object[] values)
    {
        ValidationResult result = super.validateArrayValues(values);
        
        for (Object valueToValidate : values)
        {
            result.addResult(validateSingleValue(valueToValidate));
        }
        
        return result;
    }
}
