/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.forms.repository;

import java.util.HashMap;
import java.util.Map;

import javax.jcr.Node;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.plugins.forms.question.FormQuestionDataTypeExtensionPoint;
import org.ametys.plugins.forms.question.FormQuestionType;
import org.ametys.plugins.forms.question.FormQuestionTypeExtensionPoint;
import org.ametys.plugins.repository.AmetysObjectFactory;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.data.type.RepositoryElementType;
import org.ametys.plugins.repository.jcr.DefaultTraversableAmetysObjectFactory;
import org.ametys.runtime.model.Model;
import org.ametys.runtime.model.ModelItem;

/**
 * {@link AmetysObjectFactory} for creating {@link FormQuestion}
 */
public class FormQuestionFactory extends DefaultTraversableAmetysObjectFactory
{
    private FormQuestionDataTypeExtensionPoint _formQuestionDataTypeExtensionPoint;
    private FormQuestionTypeExtensionPoint _formQuestionTypeExtensionPoint;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _formQuestionDataTypeExtensionPoint = (FormQuestionDataTypeExtensionPoint) manager.lookup(FormQuestionDataTypeExtensionPoint.ROLE);
        _formQuestionTypeExtensionPoint = (FormQuestionTypeExtensionPoint) manager.lookup(FormQuestionTypeExtensionPoint.ROLE);
    }
    
    @Override
    public FormQuestion getAmetysObject(Node node, String parentPath) throws AmetysRepositoryException
    {
        return new FormQuestion(node, parentPath, this);
    }
    
    /**
     * Retrieves the extension point holding all {@link RepositoryElementType} classes for form questions.
     * @return the extension point holding all {@link RepositoryElementType} classes for form questions.
     */
    public FormQuestionDataTypeExtensionPoint getFormQuestionDataTypeExtensionPoint()
    {
        return _formQuestionDataTypeExtensionPoint;
    }
   
    /**
     * Retrieves the extension point holding all {@link FormQuestionType} classes for form questions.
     * @return the extension point holding all {@link FormQuestionType} classes for form questions.
     */
    public FormQuestionTypeExtensionPoint getFormQuestionTypeExtensionPoint()
    {
        return _formQuestionTypeExtensionPoint;
    }
    
   /**
    * Get the form question model
    * @return the form question model
    */
    public Model getFormQuestionModel()
    {
        Map<String, ModelItem> items = new HashMap<>();
        for (String typeId : _formQuestionTypeExtensionPoint.getExtensionsIds())
        {
            FormQuestionType type = _formQuestionTypeExtensionPoint.getExtension(typeId);
            for (ModelItem modelItem : type.getModel().getModelItems())
            {
                if (!items.containsKey(modelItem.getName()))
                {
                    items.put(modelItem.getName(), modelItem);
                }
            }
        }
        
        return Model.of(
            "form.question.model.id", 
            "form.question.model.family.id", 
            items.values().toArray(new ModelItem[items.size()])
        );
    }
}
