/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.statistics;

import java.util.List;
import java.util.stream.IntStream;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.plugins.forms.repository.Form;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.PluginAware;
import org.ametys.runtime.plugins.admin.statistics.Statistics;
import org.ametys.runtime.plugins.admin.statistics.StatisticsNode;
import org.ametys.runtime.plugins.admin.statistics.StatisticsProvider;
import org.ametys.runtime.plugins.admin.statistics.StatisticsValue;

/**
 * Stats for the "new" content forms
 */
public class FormsStatisticsProvider implements StatisticsProvider, Serviceable, PluginAware
{
    
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;

    private String _id;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
    }
    
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _id = id;
    }
    
    public Statistics getStatistics()
    {
        List<Integer> formsSubmissions = _getFormsSubmissions();
        int submissions = formsSubmissions.stream().flatMapToInt(IntStream::of).sum();
        
        return new StatisticsNode(
                _id,
                new I18nizableText("plugin.forms", "PLUGINS_FORMS_STATISTICS_FORMS_LABEL"),
                "ametysicon-code-html-form",
                formsSubmissions.size(),
                List.of(
                    new StatisticsNode(
                        "submissions",
                        new I18nizableText("plugin.forms", "PLUGINS_FORMS_STATISTICS_FORMS_SUBMISSIONS_LABEL"),
                        "ametysicon-desktop-school-tool",
                        submissions,
                        List.of(
                            new StatisticsValue(
                                "max",
                                new I18nizableText("plugin.forms", "PLUGINS_FORMS_STATISTICS_FORMS_SUBMISSIONS_MAX_LABEL"),
                                "ametysicon-sort51",
                                formsSubmissions.size() > 0 ? formsSubmissions.get(formsSubmissions.size() - 1) : 0
                            ),
                            new StatisticsValue(
                                "median",
                                new I18nizableText("plugin.forms", "PLUGINS_FORMS_STATISTICS_FORMS_SUBMISSIONS_MEDIAN_LABEL"),
                                "ametysicon-maths-window-symbol-x",
                                formsSubmissions.size() > 0 ? formsSubmissions.get(formsSubmissions.size() / 2) : 0
                            )
                        ),
                        false
                    )
                ),
                true
            );
    }

    
    private List<Integer> _getFormsSubmissions()
    {
        String xpathQuery = "//element(*, ametys:form)";
        return _resolver.query(xpathQuery)
            .stream()
            .filter(Form.class::isInstance)
            .map(Form.class::cast)
            .map(Form::getEntries)
            .map(List::size)
            .sorted()
            .toList();
    }
}
