/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.workflow;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.workflow.AmetysObjectCheckRightsCondition;
import org.ametys.plugins.forms.helper.FormMailHelper;
import org.ametys.plugins.forms.helper.FormMailHelper.LimitationMailType;
import org.ametys.plugins.forms.helper.LimitedEntriesHelper;
import org.ametys.plugins.forms.repository.Form;
import org.ametys.plugins.forms.repository.FormEntry;
import org.ametys.plugins.workflow.AbstractWorkflowComponent;
import org.ametys.plugins.workflow.EnhancedFunction;
import org.ametys.plugins.workflow.support.WorkflowHelper.WorkflowVisibility;
import org.ametys.runtime.i18n.I18nizableText;

import com.opensymphony.module.propertyset.PropertySet;
import com.opensymphony.workflow.InvalidActionException;
import com.opensymphony.workflow.WorkflowException;

/**
 * OS workflow function to active the entry
 */
public class ActiveEntryFunction extends AbstractWorkflowComponent implements EnhancedFunction
{
    /** The limited entries helper */
    protected LimitedEntriesHelper _limitedEntriesHelper;
    
    /** The form mail helper */
    protected FormMailHelper _formMailHelper;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _limitedEntriesHelper = (LimitedEntriesHelper) manager.lookup(LimitedEntriesHelper.ROLE);
        _formMailHelper = (FormMailHelper) manager.lookup(FormMailHelper.ROLE);
    }
    
    @Override
    public void execute(Map transientVars, Map args, PropertySet ps) throws WorkflowException
    {
        FormEntry entry = (FormEntry) transientVars.get(AmetysObjectCheckRightsCondition.AMETYS_OBJECT_KEY);
        if (!entry.isActive())
        {
            if (_limitedEntriesHelper.isFormLimitIsReached(entry.getForm()))
            {
                List<String> i18nParams = new ArrayList<>();
                i18nParams.add(entry.getForm().getTitle());
                
                addWorkflowError(transientVars, new I18nizableText("plugin.forms", "PLUGINS_FORMS_ACTIVE_ENTRY_LIMIT_CONDITION_FAILED", i18nParams));
                
                throw new InvalidActionException("Can activate the entry because the form (" + entry.getForm().getId() + ") limit is reach");
            }
            else
            {   
                entry.setActive(true);
                Form form = entry.getForm();
                
                Optional<String[]> emailsAsArrayOpt = form.getAdminEmails();
                if (emailsAsArrayOpt.isPresent())
                {
                    String[] emailsAsArray = emailsAsArrayOpt.get();
                    if (form.isEntriesLimited())
                    {
                        int totalSubmissions = form.getActiveEntries().size();
                        Long maxEntries = form.getMaxEntries().get();
                        if (maxEntries == totalSubmissions)
                        {
                            _formMailHelper.sendLimitationReachedMailForAdmin(entry, emailsAsArray, LimitationMailType.LIMIT);
                        }
                        else if (form.isQueueEnabled() && form.getQueueSize().isPresent() && form.getQueueSize().get() + maxEntries == totalSubmissions)
                        {
                            _formMailHelper.sendLimitationReachedMailForAdmin(entry, emailsAsArray, LimitationMailType.QUEUE);
                        }
                    }
                }
            }
        }
    }

    @Override
    public FunctionType getFunctionExecType()
    {
        return FunctionType.PRE;
    }
    
    @Override
    public I18nizableText getLabel()
    {
        return new I18nizableText("plugin.forms", "PLUGINS_FORMS_ACTIVE_ENTRY_FUNCTION_LABEL");
    }
    
    @Override
    public List<WorkflowVisibility> getVisibilities()
    {
        List<WorkflowVisibility> visibilities = EnhancedFunction.super.getVisibilities();
        visibilities.add(WorkflowVisibility.USER);
        return visibilities;
    }
}
