/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.actions;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;

import org.ametys.core.cocoon.JSonReader;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.forms.dao.FormDAO;
import org.ametys.plugins.forms.dao.FormPageDAO;
import org.ametys.plugins.forms.dao.FormQuestionDAO;
import org.ametys.plugins.forms.repository.Form;
import org.ametys.plugins.forms.repository.FormPage;
import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.authentication.AccessDeniedException;

/**
 * Get the form structure to built tree
 */
public class GetFormStructureAction extends ServiceableAction
{
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The form DAO */
    protected FormDAO _formDAO;
    
    /** DAO for manipulating form pages */
    protected FormPageDAO _formPageDAO;
    
    /** DAO for manipulating form pages */
    protected FormQuestionDAO _formQuestionDAO;
    
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _formDAO = (FormDAO) serviceManager.lookup(FormDAO.ROLE);
        _formPageDAO = (FormPageDAO) serviceManager.lookup(FormPageDAO.ROLE);
        _formQuestionDAO = (FormQuestionDAO) serviceManager.lookup(FormQuestionDAO.ROLE);
        _currentUserProvider = (CurrentUserProvider) serviceManager.lookup(CurrentUserProvider.ROLE);
    }
    
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        @SuppressWarnings("unchecked")
        Map<String, Object> jsParameters = (Map<String, Object>) objectModel.get(ObjectModelHelper.PARENT_CONTEXT);
        
        AmetysObject currentNode = _resolver.resolveById((String) jsParameters.get("node"));
        
        List<Map<String, Object>> nodes = new ArrayList<>();
        if (currentNode instanceof Form form)
        {
            _checkReadAccess(form);
            for (FormPage page : form.getPages())
            {
                nodes.add(_formPageDAO.getPageProperties(page, false));
            }
        }
        else if (currentNode instanceof FormPage page) 
        {
            _checkReadAccess(page.getForm());
            for (FormQuestion question : page.getQuestions())
            {
                nodes.add(_formQuestionDAO.getQuestionProperties(question, false));
            }
        }
        
        Map<String, Object> result = new HashMap<>();
        result.put("pages", nodes);
        
        Request request = ObjectModelHelper.getRequest(objectModel);
        request.setAttribute(JSonReader.OBJECT_TO_READ, result);
        
        return EMPTY_MAP;
    }
    
    private void _checkReadAccess(Form form)
    {
        UserIdentity user = _currentUserProvider.getUser();
        if (!_formDAO.hasReadRightOnForm(user, form))
        {
            throw new AccessDeniedException("User '" + user + "' is not allowed to access the form with id '" + form.getId() + "'");
        }
    }
}
