/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.question.sources;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.JSONUtils;
import org.ametys.plugins.forms.helper.FormElementDefinitionHelper;
import org.ametys.plugins.forms.question.types.impl.ChoicesListQuestionType;
import org.ametys.plugins.forms.repository.Form;
import org.ametys.plugins.forms.repository.FormEntry;
import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.type.ModelItemTypeConstants;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.runtime.plugin.component.PluginAware;

/**
 * Static class for creating {@link ChoiceSourceType} from xml congig
 */
public abstract class AbstractSourceType extends AbstractLogEnabled implements ChoiceSourceType, Serviceable, Configurable, PluginAware
{
    /** The question param key */
    public static final String QUESTION_PARAM_KEY = "question";
    
    /** JSON helper */
    protected JSONUtils _jsonUtils;
    /** The form element definition helper */
    protected FormElementDefinitionHelper _formElementDefinitionHelper;
    
    /** Forms */
    protected String _pluginName;
    /** Id of source type */
    protected String _id;
    /** Label of source type */
    protected I18nizableText _label;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _jsonUtils = (JSONUtils) manager.lookup(JSONUtils.ROLE);
        _formElementDefinitionHelper = (FormElementDefinitionHelper) manager.lookup(FormElementDefinitionHelper.ROLE);
    }
    
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _pluginName = pluginName;
    }

    public void configure(Configuration configuration) throws ConfigurationException
    {
        _id = configuration.getAttribute("id");
        
        Configuration childLabel = configuration.getChild("label");
        _label = I18nizableText.getI18nizableTextValue(childLabel, "plugin." + _pluginName, childLabel.getValue());
    }

    public String getId()
    {
        return _id;
    }

    public I18nizableText getLabel()
    {
        return _label;
    }
    
    public List<String> getFieldToDisableIfFormPublished()
    {
        return new ArrayList<>();
    }
    
    public String getStorageType(FormQuestion question)
    {
        return ModelItemTypeConstants.STRING_TYPE_ID;
    }
    
    public String getJSRenderer()
    {
        return "Ametys.plugins.forms.helper.SearchEntriesGridHelper.renderStringChoiceList";
    }
    
    public String getJSConverter()
    {
        return null;
    }
    
    public Object removeEmptyOrOtherValue(Object value)
    {
        if (value == null)
        {
            return null;
        }
        
        if (value.getClass().isArray())
        {
            List<String> newVal = new ArrayList<>();
            for (String val : (String[]) value)
            {
                if (StringUtils.isNotBlank(val) && !val.equals(ChoicesListQuestionType.OTHER_OPTION_VALUE))
                {
                    newVal.add(val);
                }
            }
            
            if (newVal.isEmpty())
            {
                return null;
            }
            
            return newVal.toArray(new String[newVal.size()]);
        }
        else if (StringUtils.isNotBlank((String) value) && !value.equals(ChoicesListQuestionType.OTHER_OPTION_VALUE))
        {
            return value;
        }
        
        return null;
    }
    
    /**
     * Get question from params
     * @param params the params
     * @return the form question
     */
    protected FormQuestion _getQuestionFromParam(Map<String, Object> params)
    {
        return (FormQuestion) params.get(QUESTION_PARAM_KEY);
    }
    
    public Object valueToJSONForClient(Object value, FormQuestion question, FormEntry entry, ModelItem modelItem) throws Exception
    {
        return _getComputedStringValue(value, question, entry);
    }
    
    /**
     * Get computed string value for choice list
     * @param valueToJSONForClient the value for this question
     * @param question the question
     * @param entry the entry
     * @return the computed value
     * @throws Exception if an error occurred during computing
     */
    protected Object _getComputedStringValue(Object valueToJSONForClient, FormQuestion question, FormEntry entry) throws Exception
    {
        String name = question.getNameForForm();
        ChoicesListQuestionType type = (ChoicesListQuestionType) question.getType();
        
        if (entry.isMultiple(name))
        {
            List<Map<String, Object>> values = new ArrayList<>();
            if (valueToJSONForClient != null)
            {
                for (Object val : (List) valueToJSONForClient)
                {
                    Map<String, Object> transformedValue = new HashMap<>();
                    transformedValue.put("value", val);
                    
                    Map<String, Object> enumParam = new HashMap<>();
                    enumParam.put(AbstractSourceType.QUESTION_PARAM_KEY, question);
                    transformedValue.put("label", type.getSourceType(question).getEntry(new ChoiceOption(val), enumParam));
                    
                    values.add(transformedValue);
                }
            }
            
            if (type.hasOtherOption(question))
            {
                String otherValue = entry.getValue(ChoicesListQuestionType.OTHER_PREFIX_DATA_NAME + name);
                if (StringUtils.isNotBlank(otherValue))
                {
                    Map<String, Object> transformedValue = new HashMap<>();
                    transformedValue.put("value", otherValue);
                    transformedValue.put("label", otherValue);
                    
                    values.add(transformedValue);
                }
            }
            
            return values;
        }
        else
        {
            Map<String, Object> transformedValue = new HashMap<>();

            if (type.hasOtherOption(question) && StringUtils.isNotBlank(entry.getValue(ChoicesListQuestionType.OTHER_PREFIX_DATA_NAME + name)))
            {
                String otherValue = entry.getValue(ChoicesListQuestionType.OTHER_PREFIX_DATA_NAME + name);
                transformedValue.put("value", otherValue);
                transformedValue.put("label", otherValue);
            }
            else if (valueToJSONForClient != null)
            {
                transformedValue.put("value", valueToJSONForClient);
                
                Map<String, Object> enumParam = new HashMap<>();
                enumParam.put(AbstractSourceType.QUESTION_PARAM_KEY, question);
                transformedValue.put("label", type.getSourceType(question).getEntry(new ChoiceOption(valueToJSONForClient), enumParam));
            }
                
            
            return transformedValue;
        }
    }
    
    /**
     * Get computed complex value as user or content for choice list
     * @param valueToJSONForClient the value for this question
     * @param question the question
     * @param entry the entry
     * @return the computed value
     * @throws Exception if an error occurred
     */
    @SuppressWarnings("unchecked")
    protected Object _getComputedComplexValue(Object valueToJSONForClient, FormQuestion question, FormEntry entry) throws Exception
    {
        String name = question.getNameForForm();
        ChoicesListQuestionType type = (ChoicesListQuestionType) question.getType();
        
        if (!type.hasOtherOption(question))
        {
            return valueToJSONForClient;
        }
        
        if (entry.isMultiple(name))
        {
            String otherValue = entry.getValue(ChoicesListQuestionType.OTHER_PREFIX_DATA_NAME + name);
            if (StringUtils.isNotBlank(otherValue))
            {
                Map<String, Object> transformedValue = new HashMap<>();
                transformedValue.put("value", otherValue);
                transformedValue.put("isOther", true);
                
                List<Object> computedVal = new ArrayList<>();
                if (valueToJSONForClient != null)
                {
                    // Invoke an array list before because valueToJSONForClient can be an immutable map
                    computedVal.addAll((List<Object>) valueToJSONForClient);
                }
                computedVal.add(transformedValue);
                
                return computedVal;
            }
        }
        else
        {
            String otherValue = entry.getValue(ChoicesListQuestionType.OTHER_PREFIX_DATA_NAME + name);
            if (StringUtils.isNotBlank(otherValue))
            {
                Map<String, Object> transformedValue = new HashMap<>();
                transformedValue.put("value", otherValue);
                transformedValue.put("isOther", true);
                
                return transformedValue;
            }
        }
        
        return valueToJSONForClient;
    }
    
    public String value2String(Object value)
    {
        return value.toString();
    }
    
    public boolean hasRight(UserIdentity user, Form form)
    {
        // By default, no right to check to acces to source data
        return true;
    }
}
