/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * This class provides a widget to select a form in the tree.
 * 
 * This widget is registered for fields of type Ametys.form.WidgetManager#TYPE_STRING.
 */
Ext.define('Ametys.plugins.forms.widget.FormsSelector', {
    extend : 'Ametys.form.AbstractField',
    
    xtype: 'edition.forms-selector',
    
    /**
     * @cfg {String} emptyText The text for empty field
     */
    emptyText: "{{i18n PLUGINS_FORMS_WIDGET_FORM_CHOOSE_EMPTY_TEXT}}",
    
    initComponent: function()
    {
        this._text = Ext.create('Ext.Component', {
            cls: Ametys.form.AbstractField.READABLE_TEXT_CLS,
            html: this.emptyText,
            flex: 1
        });
        this._chooseButton = Ext.create('Ext.button.Button', {
            iconCls: 'ametysicon-file98',
            tooltip: "{{i18n PLUGINS_FORMS_WIDGET_FORM_CHOOSE_BUTTON}}",
            handler: this._showChoosePopup,
            scope: this
        });
        this._deleteButton = Ext.create('Ext.button.Button', {
            iconCls: 'ametysicon-delete30',
            tooltip: "{{i18n PLUGINS_FORMS_WIDGET_FORM_DELETE_BUTTON}}",
            handler: this._deleteValue,
            scope: this,
            hidden: true
        });
        
        this.items = [this._text, this._chooseButton, this._deleteButton];

        this.layout = 'hbox';
        this.cls = this.emptyCls;
        
        this.callParent(arguments);
    },
    
    afterRender: function()
    {
        this.callParent(arguments);
        this._updateUI();
    },
    
    setValue: function(value) 
    {   
        this.callParent(arguments);
        this._updateUI();
    },
    
    /**
     * Opens a popup to choose value for current widget
     * @private
     */
    _showChoosePopup: function()
    {
        var me = this;
        Ametys.plugins.forms.helper.ChooseForm.open({
            title: "{{i18n PLUGINS_FORMS_WIDGET_FORM_DIALOG_TITLE}}",
            iconCls: 'ametysicon-file98',
            helpmessage: "{{i18n PLUGINS_FORMS_WIDGET_FORM_DIALOG_HINT}}",
            callback: function(id) {
                me.setValue(id);
            }
        });
    },
    
    /**
     * Deletes the current widget value
     * @private
     */
     _deleteValue: function()
     {
         this.setValue(null);
         this._updateUI();
     },
     
     /**
     * Updates UI
     * @private
     */
    _updateUI: function()
    {   
        var value = this.value;
        
        if (!this.rendered)
        {
            return;
        }
        
        if (!value) 
        {
            this._deleteButton.hide();
            this._chooseButton.show();
        }
        else
        {
            this._deleteButton.show();
            this._chooseButton.hide();
        }
        
        this._updateDisplayField();
    }, 
    
    /**
     * Updates the display field as an understanding value for the end user
     * @private
     */
    _updateDisplayField: function()
    {
        if (!this.rendered)
        {
            return;
        }
        
        if (this.value)
        {
            Ametys.plugins.forms.dao.FormDAO.getFormFullPath([this.value], Ext.bind(this._updateDisplayFieldCb, this));
        }
        else
        {
            this._updateDisplayFieldCb(null);
        }
    },
    
    /**
     * Sets the readable value from the selected directory
     * @param {String} fullPath The form full path
     * @private
     */
    _updateDisplayFieldCb: function(fullPath)
    {
        this._readableValue = fullPath;
        this._text.update(this._getReadableValue());
    },
    
    /**
     * Transform the widget value in a human readable string
     * @return {String} a human readable string
     * @private
     */
    _getReadableValue: function ()
    {
        if (this.value)
        {
            return Ext.String.escapeHtml(this._readableValue || this.value);
        }
        else
        {
            return this.emptyText;
        }
    }
});