/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.question.autofill;

import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserManager;
import org.ametys.plugins.forms.helper.FormElementDefinitionHelper;
import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.runtime.plugin.component.PluginAware;

/**
 * Static class for creating {@link AutofillSource} from xml configuration
 */
public abstract class AbstractStaticAutoFillSource extends AbstractLogEnabled implements AutofillSource, Serviceable, Configurable, PluginAware
{
    /** Forms */
    protected String _pluginName;
    /** Id of computing type */
    protected String _id;
    /** Label of computing type */
    protected I18nizableText _label;
    /** Description of computing type */
    protected I18nizableText _description;

    /** The autofill source extension point */
    protected AutofillSourceExtensionPoint _autoFillSourceExtensionPoint;
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    /** The users manager */
    protected UserManager _userManager;
    /** The form element definition helper */
    protected FormElementDefinitionHelper _formElementDefinitionHelper;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _autoFillSourceExtensionPoint = (AutofillSourceExtensionPoint) manager.lookup(AutofillSourceExtensionPoint.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _userManager = (UserManager) manager.lookup(UserManager.ROLE);
        _formElementDefinitionHelper = (FormElementDefinitionHelper) manager.lookup(FormElementDefinitionHelper.ROLE);
    }
    
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _pluginName = pluginName;
    }

    public void configure(Configuration configuration) throws ConfigurationException
    {
        _id = configuration.getAttribute("id");
        
        Configuration childLabel = configuration.getChild("label");
        _label = I18nizableText.getI18nizableTextValue(childLabel, "plugin." + _pluginName, childLabel.getValue());
        
        Configuration childDesc = configuration.getChild("description");
        _description = I18nizableText.getI18nizableTextValue(childDesc, "plugin." + _pluginName, childDesc.getValue());
    }

    public String getId()
    {
        return _id;
    }

    public I18nizableText getLabel()
    {
        return _label;
    }
    
    public I18nizableText getDescription()
    {
        return _description;
    }
    
    public void saxAdditionalInfos(ContentHandler contentHandler, FormQuestion question) throws SAXException
    {
        String autofillValue = getAutofillValue(question);
        if (StringUtils.isNotBlank(autofillValue))
        {
            XMLUtils.createElement(contentHandler, "default-value", autofillValue);
        }
    }

}
