/*
 * Copyright 2022 Anyware Services
 * 
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.workflow;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.forms.helper.FormMailHelper;
import org.ametys.plugins.forms.repository.FormEntry;
import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.plugins.repository.AmetysObjectResolver;

/**
 * Helper component allowing to retrieve information in order to send emails
 */
public class MailInformationHelper extends AbstractMailInformationHelper
{
    /** The Avalon role */
    public static final String ROLE = MailInformationHelper.class.getName();

    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The form mail helper */
    protected FormMailHelper _formMailHelper;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _formMailHelper = (FormMailHelper) serviceManager.lookup(FormMailHelper.ROLE);
    }
    
    @Override
    protected List<MailInfo> _getMailInfos(String formId, Object entryId)
    {
        List<MailInfo> emails = new ArrayList<> ();
        
        FormEntry formEntry = _resolver.resolveById((String) entryId);
        UserIdentity userId = formEntry.getUser();
        if (userId != null)
        {
            User user = _userManager.getUser(userId);
            if (user != null)
            {
                String email = user.getEmail();
                if (StringUtils.isNotBlank(email))
                {
                    MailInfo mailInfo = new MailInfo(
                            email,
                            user.getFullName() + " <" + email + ">"
                            );
                    
                    emails.add(mailInfo);
                }
            }
        }
        
        for (FormQuestion question : _formMailHelper.getQuestionWithMail(formId))
        {
            Optional<String> receiver = _formMailHelper.getReceiver(formEntry, Optional.of(question.getNameForForm()));
            if (receiver.isPresent())
            {
                MailInfo mailInfo = new MailInfo(
                        receiver.get(),
                    question.getTitle() + " <" + receiver.get() + ">"
                );
                emails.add(mailInfo);
            }
        }
        
        return emails;
    }
}
