/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.content.data;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import javax.jcr.Node;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.content.ContentHelper;
import org.ametys.cms.repository.Content;
import org.ametys.core.cocoon.JSonReader;
import org.ametys.core.right.RightManager;
import org.ametys.core.util.DateUtils;
import org.ametys.core.util.URIUtils;
import org.ametys.plugins.forms.FormsException;
import org.ametys.plugins.forms.content.Form;
import org.ametys.plugins.forms.content.jcr.FormPropertiesManager;
import org.ametys.plugins.forms.content.table.FormTableManager;
import org.ametys.plugins.repository.AmetysObjectResolver;

/**
 * This generator is used in order to retrieve information on the forms 
 * of a given site in a given language
 */
public class GetFormsListAction extends ServiceableAction
{
    /** Component gathering methods to retrieve forms */
    private FormPropertiesManager _formPropertiesManager;
    
    /** The form data manager */
    private FormTableManager _formTableManager;
    
    /** The Ametys object resolver */
    private AmetysObjectResolver _resolver;
    
    private ContentHelper _contentHelper;

    private RightManager _rightManager;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _formPropertiesManager = (FormPropertiesManager) serviceManager.lookup(FormPropertiesManager.ROLE);
        _formTableManager = (FormTableManager) serviceManager.lookup(FormTableManager.ROLE);
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _contentHelper = (ContentHelper) serviceManager.lookup(ContentHelper.ROLE);
        _rightManager = (RightManager) serviceManager.lookup(RightManager.ROLE);
    }
    
    @Override
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        @SuppressWarnings("unchecked")
        Map<String, Object> jsParameters = (Map<String, Object>) objectModel.get(ObjectModelHelper.PARENT_CONTEXT);
        
        String siteName = (String) jsParameters.get("siteName");
        String language = (String) jsParameters.get("lang");
        
        
        if (StringUtils.isEmpty(siteName) || StringUtils.isEmpty(language))
        {
            throw new ProcessingException("The site name and language must be provided.");
        }
        
        Map<String, Object> result = new HashMap<>();
        
        List<Map<String, Object>> formsList = new ArrayList<> ();
        try
        {
            List<Node> contentNodes = _formPropertiesManager.getFormContentNodes(siteName, language);
            for (Node contentNode : contentNodes)
            {
                Content content = _resolver.resolve(contentNode, false);
                
                if (_rightManager.currentUserHasReadAccess(content))
                {
                    // Get the corresponding forms
                    List<Form> forms = _formPropertiesManager.getForms(contentNode);
                    for (Form form : forms)
                    {
                        Map<String, Object> formMap = new HashMap<> ();
                        try
                        {                        
                            List<UserEntry> entries = _formTableManager.getSubmissions(form, new HashMap<>(), 0, Integer.MAX_VALUE, null);
                            int totalSubmissions = entries.size();
                            Date lastSubmissionDate = _getLastSubmissionDate(entries);
                            
                            formMap.put("id", form.getId());
                            formMap.put("label", URIUtils.decode(form.getLabel()));
                            formMap.put("contentName", content.getName());
                            formMap.put("contentId", content.getId());
                            formMap.put("contentTitle", _contentHelper.getTitle(content));
                            formMap.put("authorFullName", content.getCreator());
                            
                            formMap.put("entriesAmount", String.valueOf(totalSubmissions));
                            formMap.put("lastEntry", lastSubmissionDate != null ? DateUtils.dateToString(lastSubmissionDate) : null);
                            formMap.put("workflowName", Objects.toString(form.getWorkflowName(), StringUtils.EMPTY));
                            
                            formsList.add(formMap);
                        }
                        catch (FormsException e)
                        {
                            getLogger().error("Failed to get SQL table for form '" + form.getId() + "' for content of id '" + content.getId() + "'", e);
                        }
                    }
                }
            }
            
            result.put("forms", formsList);
        }
        catch (Exception e)
        {
            throw new ProcessingException("An error occurred while processing the forms list.", e);
        }
        
        Request request = ObjectModelHelper.getRequest(objectModel);
        request.setAttribute(JSonReader.OBJECT_TO_READ, result);
        return EMPTY_MAP;
    }
    
    /**
     * Get the date of the most recent {@link UserEntry}
     * @param entries the list of entries
     * @return the date of the most recent entry
     */
    private Date _getLastSubmissionDate (List<UserEntry> entries)
    {
        Date lastSubmissionDate = null;
        if (entries.size() > 0)
        {
            lastSubmissionDate = entries.get(0).getCreationDate();
            for (UserEntry entry : entries)
            {
                Date creationDate = entry.getCreationDate();
                if (lastSubmissionDate.before(creationDate))
                {
                    lastSubmissionDate = creationDate;
                }
            }
        }
        
        return lastSubmissionDate;
    }
}
