/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms;

import java.time.Period;
import java.time.ZonedDateTime;
import java.util.Map;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.core.trace.ForensicLogger;
import org.ametys.core.user.population.UserPopulationDAO;
import org.ametys.core.user.status.PersonalDataPolicy;
import org.ametys.core.user.status.UserStatusInfo;
import org.ametys.plugins.forms.repository.FormEntry;
import org.ametys.plugins.forms.repository.FormEntryFactory;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.query.QueryHelper;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.UserExpression;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Anonymize all the answers given by the user after a given delay since he is missing
 */
public class FormEntryPersonalDataPolicy extends AbstractLogEnabled implements PersonalDataPolicy, Serviceable, Initializable
{
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    private Period _delay;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
    }
    
    public void initialize() throws Exception
    {
        Long config = Config.getInstance().<Long>getValue("form.data.policy.form.entries.retention", false, null);
        _delay = config != null && config >= 0 ? Period.ofMonths(config.intValue()) : null;
    }
    
    public AnonymizationResult process(UserStatusInfo userStatusInfo)
    {
        if (_delay != null && ZonedDateTime.now().isAfter(userStatusInfo.getMissingSinceDate().plus(_delay)))
        {
            Expression userExpression = new UserExpression(FormEntry.ATTRIBUTE_USER, Operator.EQ, userStatusInfo.getUserIdentity());
            String query = QueryHelper.getXPathQuery(null, FormEntryFactory.NODE_TYPE, userExpression);
            try (AmetysObjectIterable<FormEntry> entries = _resolver.query(query))
            {
                if (entries.getSize() > 0)
                {
                    for (FormEntry entry : entries)
                    {
                        entry.anonymize();
                    }
                    ForensicLogger.info("data.policy.gdpr.anonymize.form.submissions", Map.of("handled", Long.toString(entries.getSize()), "identity", userStatusInfo.getUserIdentity()), UserPopulationDAO.SYSTEM_USER_IDENTITY);
                    return AnonymizationResult.PROCESSED;
                }
                else
                {
                    return AnonymizationResult.NO_DATA;
                }
            }
        }
        else
        {
            return AnonymizationResult.TOO_EARLY;
        }
    }

}
