/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.content.actions;

import java.util.HashMap;
import java.util.Map;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;

import org.ametys.cms.repository.Content;
import org.ametys.core.cocoon.ActionResultGenerator;
import org.ametys.core.cocoon.JSonReader;
import org.ametys.core.right.RightManager;
import org.ametys.plugins.forms.FormsException;
import org.ametys.plugins.forms.content.jcr.FormPropertiesManager;
import org.ametys.plugins.forms.content.table.FormTableManager;
import org.ametys.runtime.authentication.AccessDeniedException;

/**
 * Get the number of submissions for one calendar for a form
 */
public class GetSubmissionAction extends ServiceableAction
{
    /** Form table manager. */
    protected static FormTableManager _formTableManager;
    /** Form properties manager. */
    protected FormPropertiesManager _formPropertiesManager;
    /** Rights manager */
    protected RightManager _rightManager;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        _formTableManager = (FormTableManager) smanager.lookup(FormTableManager.ROLE);
        _formPropertiesManager = (FormPropertiesManager) smanager.lookup(FormPropertiesManager.ROLE);
        _rightManager = (RightManager) smanager.lookup(RightManager.ROLE);
        super.service(smanager);
    }
    
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Map<String, Object> result = new HashMap<>();
        Request request = ObjectModelHelper.getRequest(objectModel);
        String formId = request.getParameter("formId");
        Content content = _formPropertiesManager.getFormContent(formId);

        if (!_rightManager.currentUserHasReadAccess(content))
        {
            throw new AccessDeniedException("Access denied to form " + formId);
        }
        
        int submissions = getTotalSubmissions(formId);
        result.put("submissions", submissions);
        result.put("success", true);
        request.setAttribute(JSonReader.OBJECT_TO_READ, result);
        request.setAttribute(ActionResultGenerator.MAP_REQUEST_ATTR, result);
        return EMPTY_MAP;
    }


    /**
     * Get the total count of submissions for a form.
     * @param formId the id of the form
     * @return the total count of submissions
     * @throws FormsException if an error occurs.
     */
    public static int getTotalSubmissions(String formId) throws FormsException
    {
        return _formTableManager.getTotalSubmissions(formId);
    }
}
