/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.content.workflow;

import java.io.IOException;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.SAXException;

import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.forms.FormsException;
import org.ametys.plugins.forms.content.Form;
import org.ametys.plugins.forms.content.data.FieldValue;
import org.ametys.plugins.forms.content.data.UserEntry;
import org.ametys.plugins.forms.content.processing.AbstractFormFieldGenerator;
import org.ametys.plugins.forms.content.table.FormTableManager;
import org.ametys.runtime.authentication.AccessDeniedException;

/**
 * Generate the entry of a form
 */
public class FormEntryInformationGenerator extends AbstractFormFieldGenerator
{
    /** The form data manager. */
    protected FormTableManager _formTableManager;
    
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _formTableManager = (FormTableManager) smanager.lookup(FormTableManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);
    }
    
    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);

        String siteName = (String) request.get("siteName");
        String formId = (String) request.get("formId");
        String entryId = (String) request.get("entryId");
        
        contentHandler.startDocument();
        
        try
        {
            Form form = _formPropManager.getForm(siteName, formId);
            if (form == null)
            {
                throw new ProcessingException("The form of ID '" + formId + " can't be found for site '" + siteName + "'.");
            }
            
            Map<String, FieldValue> columns = _formTableManager.getColumns(form);
            UserEntry entry = _formTableManager.getSubmission(form, columns, Integer.parseInt(entryId));
            
            UserIdentity currentUser = _currentUserProvider.getUser();
            if (!entry.getUserIdentity().equals(currentUser))
            {
                throw new AccessDeniedException("User '" + currentUser + "' is not allowed to access to user entry data.");
            }
            
            AttributesImpl atts = new AttributesImpl();
            atts.addCDATAAttribute("id", entryId);
            atts.addCDATAAttribute("formId", formId);
            atts.addCDATAAttribute("siteName", siteName);
            
            XMLUtils.startElement(contentHandler, "entry", atts);
            for (FieldValue fdValue : entry.getValues())
            {
                saxFieldValue(fdValue);
            }
            XMLUtils.endElement(contentHandler, "entry");
        }
        catch (FormsException | SAXException e) 
        {
            getLogger().error("An error occurred saxing entry information for form '"  + formId + "' and entry '" + entryId + "'", e);
        }
        
        contentHandler.endDocument();
    }

}
