/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.helper;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.environment.Request;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.observation.Event;
import org.ametys.core.observation.ObservationManager;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.forms.FormEvents;
import org.ametys.plugins.forms.dao.FormDAO;
import org.ametys.plugins.forms.dao.FormEntryDAO;
import org.ametys.plugins.forms.repository.Form;
import org.ametys.plugins.forms.repository.FormEntry;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.plugins.repository.query.expression.UserExpression;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * The helper to handler limited entries
 */
public class LimitedEntriesHelper extends AbstractLogEnabled implements Serviceable, Component
{
    /** Avalon ROLE. */
    public static final String ROLE = LimitedEntriesHelper.class.getName();
    
    /** Ametys object resolver. */
    protected AmetysObjectResolver _resolver;
    
    /** The form mail helper */
    protected FormMailHelper _formMailHelper;
    
    /** The form DAO */
    protected FormDAO _formDAO;
    
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;

    /** Observer manager. */
    protected ObservationManager _observationManager;
    
    /** The form entry DAO */
    protected FormEntryDAO _formEntryDAO;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _formMailHelper = (FormMailHelper) manager.lookup(FormMailHelper.ROLE);
        _formDAO = (FormDAO) manager.lookup(FormDAO.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _observationManager = (ObservationManager) manager.lookup(ObservationManager.ROLE);
        _formEntryDAO = (FormEntryDAO) manager.lookup(FormEntryDAO.ROLE);
    }
    
    /**
     * Get the form properties relevant for limiting entries number
     * @param formId Id of the current form
     * @return a map of the form properties 
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> getLimitedEntriesProperties(String formId)
    {
        Map<String, Object> limitProperties = new HashMap<> ();
        
        Form form = _resolver.resolveById(formId);
        _formDAO.checkHandleFormRight(form);
        
        limitProperties.put(Form.LIMIT_TO_ONE_ENTRY_BY_USER, form.isLimitedToOneEntryByUser());
        limitProperties.put(Form.LIMIT_ENTRIES_ENABLED, form.isEntriesLimited());
        _addOptionalProperty(Form.MAX_ENTRIES, form.getMaxEntries(), limitProperties, null);
        _addOptionalProperty(Form.REMAINING_MSG, form.getRemainingMessage(), limitProperties, null);
        _addOptionalProperty(Form.CLOSED_MSG, form.getClosedMessage(), limitProperties, null);
        
        limitProperties.put(Form.QUEUE_ENABLED, form.isQueueEnabled());
        _addOptionalProperty(Form.QUEUE_SIZE, form.getQueueSize(), limitProperties, null);
        _addOptionalProperty(Form.QUEUE_CLOSED_MSG, form.getClosedQueueMessage(), limitProperties, null);
        _addOptionalProperty(Form.QUEUE_SENDER, form.getQueueMailSender(), limitProperties, Config.getInstance().getValue("smtp.mail.from"));
        _addOptionalProperty(Form.QUEUE_RECEIVER, form.getQueueMailReceiver(), limitProperties, FormMailHelper.RECEIVER_COMBOBOX_ENTRY_USER_VALUE);
        _addOptionalProperty(Form.QUEUE_SUBJECT, form.getQueueMailSubject(), limitProperties, null);
        _addOptionalProperty(Form.QUEUE_BODY, form.getQueueMailBody(), limitProperties, null);
        
        return limitProperties;
    }
    
    private void _addOptionalProperty(String propertyName, Optional<? extends Object> value, Map<String, Object> limitProperties, Object defaultValue)
    {
        if (value.isPresent())
        {
            limitProperties.put(propertyName, value.get());
        }
        else if (defaultValue != null)
        {
            limitProperties.put(propertyName, defaultValue);
        }
    }
    
    /**
     * Set the form properties relevant for limiting entries number
     * @param limitParameters the limit parameters
     * @return the map of results
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> setEntriesLimitProperties (Map<String, Object> limitParameters)
    {
        Map<String, Object> result = new HashMap<>();
        
        Form form = _resolver.resolveById((String) limitParameters.get("formId"));
        _formDAO.checkHandleFormRight(form);
        
        form.limitToOneEntryByUser((boolean) limitParameters.get(Form.LIMIT_TO_ONE_ENTRY_BY_USER));
        if ((boolean) limitParameters.get(Form.LIMIT_ENTRIES_ENABLED))
        {
            form.limitEntries(true);
            form.setMaxEntries(((Integer) limitParameters.get(Form.MAX_ENTRIES)).longValue());
            form.setRemainingMessage((String) limitParameters.get(Form.REMAINING_MSG));
            form.setClosedMessage((String) limitParameters.get(Form.CLOSED_MSG));
            
            if ((boolean) limitParameters.get(Form.QUEUE_ENABLED))
            {
                form.enableQueue(true);
                Object queueSize = limitParameters.get(Form.QUEUE_SIZE);
                if (queueSize != null)
                {
                    form.setQueueSize(((Integer) queueSize).longValue());
                }
                else
                {
                    form.removeValue(Form.QUEUE_SIZE);
                }
                form.setClosedQueueMessage((String) limitParameters.get(Form.QUEUE_CLOSED_MSG));
                form.setQueueMailtReceiver((String) limitParameters.get(Form.QUEUE_RECEIVER));
                form.setQueueMailSender((String) limitParameters.get(Form.QUEUE_SENDER));
                form.setQueueMailSubject((String) limitParameters.get(Form.QUEUE_SUBJECT));
                form.setQueueMailBody((String) limitParameters.get(Form.QUEUE_BODY));
            }
            else
            {
                form.enableQueue(false);
            }
        }
        else
        {
            form.limitEntries(false);
        }
        form.saveChanges();

        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put("form", form);
        _observationManager.notify(new Event(FormEvents.FORM_MODIFIED, _currentUserProvider.getUser(), eventParams));
        
        return result;
    }
    
    /**
     * <code>true</code> if the user can submit the form depending of the form limitation
     * @param form the form
     * @param user the user. Can be null if the form is anonymous
     * @param clientIp the client ip of the user
     * @return <code>true</code> if the user can submit the form depending of the form limitation
     */
    public boolean canUserSubmit(Form form, UserIdentity user, String clientIp)
    {
        // Check if the form is limited to one entry by user and if the user has already answer
        if (form.isLimitedToOneEntryByUser() && hasUserAlreadyAnswer(form, user, clientIp))
        {
            return false;
        }
        
        // Check if the number of entries is limited
        Optional<Long> maxEntriesOpt = form.getMaxEntries();
        if (form.isEntriesLimited() && maxEntriesOpt.isPresent())
        {
            Optional<Long> queueSize = form.getQueueSize();
            int entriesSize = form.getActiveEntries().size();
            // If there are a queue ...
            if (form.isQueueEnabled())
            {
                // ... return true if the queue has no limit 
                // or check the entries size
                return queueSize.isEmpty() || entriesSize < (maxEntriesOpt.get() + queueSize.get());
            }
            else
            {
                // Just check the entries size with the limit of entries
                return entriesSize < maxEntriesOpt.get();
            }
        }
        
        // Return true if no limitation
        return true;
    }
    
    /**
     * <code>true</code> if the user has already answer to the form
     * @param form the form
     * @param user the user. Can be null if the form is anonymous
     * @param clientIp the client ip of the user
     * @return <code>true</code> if the user has already answer to the form
     */
    public boolean hasUserAlreadyAnswer(Form form, UserIdentity user, String clientIp)
    {
        Expression additionalQuery = user != null
            ? new UserExpression(FormEntry.ATTRIBUTE_USER, Operator.EQ, user)
            : StringUtils.isNotBlank(clientIp) 
                ? new StringExpression(FormEntry.ATTRIBUTE_IP, Operator.EQ, clientIp)
                : null;
        
        return !_formEntryDAO.getFormEntries(form, false, additionalQuery, List.of()).isEmpty();
    }
    
    /**
     * Get a forwarded client IP address if available.
     * @param request The servlet request object.
     * @return The HTTP <code>X-Forwarded-For</code> header value if present,
     * or the default remote address if not.
     */
    public String getClientIp(Request request)
    {
        String ip = "";
        String forwardedIpHeader = request.getHeader("X-Forwarded-For");
        
        if (StringUtils.isNotEmpty(forwardedIpHeader))
        {
            String[] forwardedIps = forwardedIpHeader.split("[\\s,]+");
            String forwardedIp = forwardedIps[forwardedIps.length - 1];
            if (StringUtils.isNotBlank(forwardedIp))
            {
                ip = forwardedIp.trim();
            }
        }
        
        if (StringUtils.isBlank(ip))
        {
            ip = request.getRemoteAddr();
        }
        return ip;
    }
    
    /**
     * <code>true</code> if the entry is in queue
     * @param entry the entry
     * @return <code>true</code> if the entry is in queue
     */
    public boolean isInQueue(FormEntry entry)
    {
        Form form = entry.getForm();
        return form.isQueueEnabled() && _getQueue(form).contains(entry);
    }
    
    /**
     * Deactivate an entry handling the form limitation
     * @param entryId the entry id
     */
    public void deactivateEntry(String entryId)
    {
        FormEntry entryToDeactivate = _resolver.resolveById(entryId);
        if (entryToDeactivate.isActive())
        {
            Form form = entryToDeactivate.getForm();
            if (form.isQueueEnabled())
            {
                List<FormEntry> queue = _getQueue(form);
                // The entry is on the main list
                if (!queue.isEmpty() && !queue.contains(entryToDeactivate))
                {
                    _formMailHelper.sendOutOfQueueMail(form, queue.get(0));
                }
            }
            
            // Deactivate the entry after calculating queue
            entryToDeactivate.setActive(false);
        }
    }
    
    /**
     * Get the waiting list sort by the submiting date
     * @param form the form
     * @return the waiting list
     */
    protected List<FormEntry> _getQueue(Form form)
    {
        List<FormEntry> queuedEntries = new ArrayList<>();
        
        Optional<Long> maxEntries = form.getMaxEntries();
        List<FormEntry> activeEntries = form.getActiveEntries();
        Collections.sort(activeEntries, Comparator.comparing(e -> e.getSubmitDate()));
        
        for (int i = maxEntries.get().intValue(); i < activeEntries.size(); i++)
        {
            queuedEntries.add(activeEntries.get(i));
        }
        
        return queuedEntries;
    }
    
    /**
     * <code>true</code> if the form limit is reach
     * @param form the form
     * @return <code>true</code> if the form limit is reach
     */
    public boolean isFormLimitIsReached(Form form)
    {
        Optional<Long> maxEntries = form.getMaxEntries();
        if (form.isEntriesLimited() && maxEntries.isPresent())
        {
            List<FormEntry> activeEntries = form.getActiveEntries();
            return activeEntries.size() >= maxEntries.get();
        }
        
        return false;
    }
}
